'''

   	This file is part of the Stellar Seimic Indices pipeline

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

	Copyright (C) 2016 by R. Samadi 
'''

import numpy as np
import math
import matplotlib.pyplot as plt
import ssilib
import scipy.optimize

"""
INPUT
deltanu : large separation [muHz]
numax	: peak frequency [muHz]
Ampli	: Amplitude of the gaussian envelope [ppm^2/muHz]
f	: frequency vector [muHz]

OUTPUT
A	: synthetic spectra

"""

# Ref. M13 : Mosser et al 2013, SF2A
numaxref= 3104.

# sun cutt-off freq.
nucsun = 5300.


def mixed_modes(nnup,nup,dnu,DPI,q,epsg=0.,verbose=False):
    '''
    Computed the mixed mods frequencies, periods, and other characteristics
    
    (num,Pm,Pg,nm,ng,zeta) = mixed_modes(nnup,nup,dnu,DPI,q,epsg=0.,verbose=False)
     
     nnup : radial order of the pure l=1 mode
     nup : frequency of the pure l=1 mode [muHz]
     dnu : asymptotic large separation [muHz]
     DPI : asymptotic period spacing [s]
     q   : coupling coefficient
     epsg:
    
     return (num,Pm,Pg,nm,ng,zeta)
     num : mixe modes freq. [muHz]
     Pm : mixed mode periods [s]
     Pg : associated pur g-mode period [s]
     nm : mixed mode radial orders
     ng : g mode raidal orders
     zeta : (Icore/I) where I is the mode inertia
     
   ''' 


    def mixed_mode_period_find(nup,q,DPI,epsg,nu1,nu2,eps=1e-5):
	    i = 0
	    P = 1e6/nu2
	    P1 = 1e6/nu1
	    roots = []
	    x = nup +  (dnu/math.pi)*math.atan(q*math.tan(math.pi*(P/DPI-epsg)) ) - 1e6/P
	    while ( (P <= P1) ) :
		    P0 = P * (1. + eps )
		    y = nup +  (dnu/math.pi)*math.atan(q*math.tan(math.pi*(P0/DPI-epsg)) ) - 1e6/P0
		    if ( x*y < 0):
			    # we refine the solution
			    Pr = P - x* eps*P/(y-x)
			    yn = nup +  (dnu/math.pi)*math.atan(q*math.tan(math.pi*(Pr/DPI-epsg)) ) - 1e6/Pr
#			    nu0 = 1./(1./nur - DPI/10.*1e-6)
			    if( math.fabs(yn) < 1e-5*nup):
##				    print Pr,P, P0,x,y,yn
				    roots.append(Pr)
		    P = P0
		    x = y
	    return np.array(roots)[::-1]
            

##    num = mixed_mode_freq_find(nup,q,DPI,epsg,nup-dnu/2.,nup + dnu/2.)
##    Pm = 1e6/num # mixed-mode periods (s)

    Pm= mixed_mode_period_find(nup,q,DPI,epsg,nup-dnu/2.,nup + dnu/2.)
    Nm =len(Pm)

    if(Nm <=0):
        return  (None,None,None,None,None,None) 

    num = 1e6/Pm


    
    alpha0 = dnu*1e-6*DPI # Eq. A24 Goupil et al 2013
    
    ngp =  -int(math.floor(1e6/DPI/(nup) - 0.5 - epsg)) # radial order of the g-mode close to the pure p-mode (nup)
    Pgp = (- ngp + 0.5+ epsg)*DPI # period of this g-mode
    if(verbose):
	    print 'nup,np,ngp,Pg:',nup,nnup,ngp,Pgp
    
    i = np.argmin( np.abs(num-nup) )
    ng = ngp + (np.arange(Nm)-i) # g-mode orders
    Pg = (- ng +0.5 + epsg )*DPI # pure g-mode periods [s]
    nm  = ng + nnup  # mixed-mode orders

    #  zeta = Icore/I,  Goupil et al 2013
    chi = 2.*(num/dnu)*np.cos(math.pi/DPI/num*1e6)
    zeta = 1./(1. + alpha0 *chi**2 ) #  Eq. A27-A28 Goupil et al 2013
    if (verbose):
	    for i in range(Nm):
		     print 'num,Pm,Pg,ng,nm,zeta:', num[i],Pm[i],Pg[i],ng[i],nm[i],zeta[i]

    return (num,Pm,Pg,nm,ng,zeta)
        

def universal_pattern_modes(df, delta_nu, numax, Ampli, norders = None, verbose = False , teff=4800. , DPI = -1. , q = 0.):
    
## 	# Parameters from  Mosser et al. 2011 A&A 525,L9)
## 	# eps: offset
## 	eps_Al = 0.634
## 	eps_Bl = 0.546
## 	eps = eps_Al+eps_Bl*math.log10(delta_nu)

## 	# d0l: small separation (l = angular degree)
## 	d01_Al = -0.056
## 	d01_Bl = -0.002
## 	d01 = d01_Al+d01_Bl*math.log10(delta_nu)

## 	d02_Al = 0.131
## 	d02_Bl = -0.033
## 	d02 = d02_Al+d02_Bl*math.log10(delta_nu)

## 	d03_Al = 0.280
## 	d03_Bl = 0.
## 	d03 = d03_Al+d03_Bl*math.log10(delta_nu)


## 	alphal=(0.008,0.003,0.005,0.005)

#  -----------------------------------------------------------------

# Values given by Mathieu Vrard:

        eps = 0.583 + 0.522*math.log10(delta_nu) + delta_nu*0.0091

        d01 = -0.056 - 0.002*math.log10(delta_nu)
        d02 = 0.154 - 0.058*math.log10(delta_nu)
        d03 = 0.280  

        # Values from Mosser et al (2013)
        alpha0 = 0.015*(delta_nu)**(-0.32)
        alphal = (alpha0,alpha0,alpha0,alpha0) 



	d0l=(0.,d01,d02,d03)

        
	# Mode linewidths
	if ( (teff >= 5300) & (teff <=6400) & (numax > 600.) ):
	    # Appourchaux et al (2012), table 2, at maximum mode height
	    gamma0 = 0.20 # [muHz]
	    gamma1 = 0.97  # [muHz]
	    gamma_slope = 13.0
	    gamma = gamma0 + gamma1*(teff/5777.)**gamma_slope

	else:
	    # Belkacem, sf2a, 2012
	    gamma = 0.19*(teff/4800.)**(10.8) #  *((numax/100.)*sqrt(teff/4800.) )**(-0.3)

	# Mode visibility (ref. Mosser et al. 2012 A&A, 537, A30) 
	# Visibility for pure p modes in a RGB with Teff=4800 K  (Tab. 4)
	V1 = 1.35 
	V2 = 0.64
	V3 = 0.071
	V = (1.,V1,V2,V3)

	nmax = int(round(numax/delta_nu-eps))	# Order n for which we obtain the maximum of oscillations
	nuc = nucsun*(numax/numaxref) 		# cut-off frequency

        if( norders == None):
            n1 = 1
            norders = int(nuc/delta_nu)		# Total number of the order n of the serie
            n2 = norders + 1
        else:
            n1 = nmax - norders/2 
            n2 = nmax +  norders/2 


	Nbr_l = 4			# Total number of the degree l of the serie

        L = np.array([]) # l values
        N = np.array([]) # radial order
        G = np.array([]) # line-widths
	H = np.array([]) # mode heights
        F = np.array([]) # mode frequencies

	# The mode profiles are modulated by a gaussian envelope. For Gaussian parameters, see Mosser et al. (2012 A&A 537 A30)
	width = 0.66*(numax**0.88)		# dnu_env = FWHM of the excess power envelope, see: Mosser et al. (2012 A&A 537 A30) - Table 2

	if (verbose):
	    print ("numax= %f, nuc= %f, nnmax= %i, nnuc= %i") % (numax,nuc,nmax,norders)
	if (verbose):
	    print "n   l   nu [muHz]"
	for n in range(n1,n2+1):  # Loop over radial orders n
		for l in range(0,Nbr_l):  			# Loop angular degree l
		    nu = (n + (l/2.) + eps - d0l[l] + (alphal[l]/2.) * (n-numax/delta_nu )**2) * delta_nu	#in [muHz]	#Mosser et al. 2011
		    if (verbose):
			    print ("n,l,nu: %i %i %f") % (n,l,nu)
		    if ( (l ==1) & (DPI >0.) ):
			    # we include mixe-modes (dipolar modes only)
			    (num,Pm,Pg,nm,ng,zeta) = mixed_modes(n,nu,delta_nu,DPI,q,epsg=0.,verbose=verbose)
			    i = 0 
			    for nu in num: # Loop over the mixed-modes
				    gammam = gamma*(1-zeta[i])
				    if (verbose):
					    print ("nm, num, gammam: %i %f %f") % (nm[i],nu,gammam)
				    if ( gammam > df*2.): # resolved profile
					    h =  1.
				    else: # un-resolved profile
					    h = (math.pi*gammam/2./df)
				    i = i +1
                                    F=np.append(F,nu)
                                    L=np.append(L,l)
                                    G=np.append(G,gammam)
                                    N=np.append(N,n)
                                    H=np.append(H,V[l]*h*ssilib.gaussenvelop(nu,Ampli,numax,width))
		    else:
                        
			    if ( gamma > df*2.): # resolved profile
				    h =  1.  # Lorentzian : modes shape
			    else: # un-resolved profile
				    h = (math.pi*gamma/2./df)
                            F=np.append(F,nu)
                            L=np.append(L,l)
                            G=np.append(G,gamma)
                            N=np.append(N,n)
                            H=np.append(H,V[l]*h*ssilib.gaussenvelop(nu,Ampli,numax,width))

	return (F,L,N,H,G) 


def universal_pattern(delta_nu, numax, Ampli, f, nyquist, norders = None, verbose = False, teff=4800., DPI = -1., q = 0.):
        

	# mode vector
	A=np.zeros(len(f))

	if (len(f) > 1):
		df = f[1]-f[0]	# Frequency resolution
	else:
		df = 1e-5

        (F,L,N,H,G)  = universal_pattern_modes(df, delta_nu, numax, Ampli, norders = norders ,  verbose = verbose , teff=teff , DPI = DPI , q = q)

        m = len(F)

	if (verbose):
	    print "n   l   nu [muHz]"
            
	for k in range(m):  # Loop over radial orders n
                nu =  F[k]	#  freq.
                n = N[k] #  order
                l = L[k] # angular degree l
                gamma = G[k] # line-widths
                h = H[k] # height

                if (verbose):
                        print ("n,l,nu,h,g: %i %i %f %f %f") % (n,l,nu,h,gamma)

                if ( gamma > df*2.): # resolved profile => Lorentzian : modes shape
                    Pr = 1./ ( 1. + ( 2.*(f-nu)/gamma)**2 ) #+ 1./ ( 1. + ( 2.*( (2.*nyquist - f) - nu )/gamma)**2 )
                else: # un-resolved profile => sinc shape
                    Pr = ( np.sinc( math.pi*(f-nu)/df) )**2 #+ ( np.sinc( math.pi*( (2.*nyquist - f)-nu )/df) )**2

                A = A+ h*Pr


	return A 
	
	
	




