'''

   	This file is part of the Stellar Seimic Indices pipeline

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

	Copyright (C) 2016 by R. Samadi 
'''
import numpy as np
import math

def gaussenvelop(nu,A,nu0,width):
	# width: full width at half maximum (delta_env)
        return A*np.exp(-((nu-nu0)**2)/(width**2/(4*np.log(2)))) 
        
def plf(nu,h,tau,alpha):
    # Pseudo Lorentzian Function (plf)
    # nu:  frequencies 
    # h: height
    # tau : characteristic time
    # alpha : slope
    return h/(1. + (2.*math.pi*tau *nu)**alpha)

def pef(nu,h,tau,alpha):
    # Pseudo Exponential Function 
    # nu:  frequencies 
    # h: height
    # tau : characteristic time
    return h*math.log(2.)*np.exp(- (np.abs(math.log(2.)*math.pi*tau*nu) )**alpha)


def eft(tau,alpha,func,doplot=False):
    # Given a input function (func), compute the associated e-folding time (eft)
    
    N = 2l**19
    Nf = N/2  + 1
    sampling =  tau*1e-3
    dnu = 1./(sampling*N)
    nu = np.arange(Nf) * dnu

    spc = func(nu,1.,tau,alpha)
    acf = np.fft.irfft(spc,n=N)
    acf = acf[1:N/2]/acf[0]
    time = (np.arange(N/2-1)+1.)*sampling/tau
    eft = np.interp(math.exp(-1),acf[::-1],time[::-1])*tau

    if (doplot):
	import matplotlib.pyplot  as plt
        acfth = np.exp( - np.abs(time))
        plt.figure(0)
        plt.clf()
        plt.plot(nu,spc)
        plt.loglog()

        plt.figure(1)
        plt.clf()
        plt.plot(time,acf)
        plt.plot(time,acfth)
        plt.plot([time[0],time[-1]],[math.exp(-1),math.exp(-1)])
        plt.axis(ymin=1e-3,ymax=1)
        plt.loglog()

        plt.figure(2)
        plt.clf()
        plt.plot(time,acf)
        plt.plot(time,acfth)
        plt.plot([0.,time[-1]],[math.exp(-1),math.exp(-1)])
        plt.axis(xmin=0,xmax=10.)
        plt.show()

    return eft


def eft_plt_test():
    tau=100.
    alpha = 2.

    eft = eft_plf(tau,alpha,doplot=True)

    print eft,(eft/tau-1.)


def sigma2_test():
    import matplotlib.pyplot  as plt
    lognu = -2. + np.arange(10000)/(1e4-1.)*5.
    nu = 10.**(lognu)
    tau = 1./(2.*math.pi)
    alpha = 3.2
    h = 1.
    p = plf(nu,h,tau,alpha)
    print np.trapz(p,x=nu)
    print 0.5*h/(tau*alpha*math.sin(math.pi/alpha))
    plt.figure(0)
    plt.clf()
    plt.plot(nu,p)
    plt.loglog()
    plt.show()


def rebin1d(array,n):
	nr=int(float(array.shape[0])/float(n))
	return (np.reshape(array,(n,nr))).sum(1)
