#!/usr/bin/env python
'''

   	This file is part of the Stellar Seimic Indices pipeline

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

	Copyright (C) 2016 by R. Samadi 
'''
import numpy as np
import math
from Struc.Record import Record
from Struc.Indice import Indice
import GestionFiles.GF as GF
from xml.dom import minidom
import getopt
import os,sys, traceback, signal
import datetime
import re
import multiprocessing
import os
import commands
import time

__version__ = "0.32"



def select_mle(name,data,origin,teffvalues):
    # data[:,i]:
    #  0 : ID
    #  1 : status
    #  2 : l = - log (likelihood)
    #  3 : flag of the free parameters (the ith parameter is free if the ith bit is 1)
    #  4 : white noise [ppm^2/muHz]
    #  5 : white noise err
    #  6 : log height gran [ppm^2/muHz]
    #  7 : log height gran err
    #  8 : log tau gran [10^6 s]
    #  9 : log tau gran err
    # 10 : slope gran
    # 11 : slope gran err
    # 12 : log height up [ppm^2/muHz]
    # 13 : log height up err 
    # 14 : numax up [muHz]
    # 15 : numax up  err
    # 16 : deltanu up [muHz]
    # 17 : deltanu up err
    # 18 : log height act [ppm^2/muHz]
    # 19 : log height act err
    # 20 : log tau act [10^6 s]
    # 21 : log tau act err
    # 22 : slope act
    # 23 : slope act err
    # 24 : Hessian matrix condition number
    # 25 : log taue gran [10^6 s]
    # 26 : log taue gran err
    # 27 : Amax
    # 28 : numax ACF [muHz]
    # 29 : numax ACF err
    # 30 : delta nu ACF  [muHz]
    # 31 : delta ACF  err 
    # 32 : Amax 2
    # 33 : numax ACF 2 [muHz]
    # 34 : numax ACF 2 err 
    # 35 : delta nu ACF 2 [muHz]
    # 36 : delta ACF 2 err 
    # 37 : Bmax [ppm^2/muHz]
    # 38 : log sigma2_gran [ppm^2]
    # 39 : log sigma2_gran err
    # 40 : number of objects with Activity 
    # 41 : number of objects without Activity 
    # 42 : number of objects whose the Activity is not sure
    # 43 : number of objects refit without Activity
    # 44 : number of objects fitted at the 2sd step
    # 45 : number of objects mapped at the 2sd step
    # 46 : step2_step1
    # 47 : resolution frequentiel
    # 48 : nyquist freq.

    n = data.shape[0]
    nyquist = data[:,48]

    Nbr_jour = 1./data[:,47]*1e6*1/86400.	

    # Status
    #status=3, probleme dans indices sismiques et/ou granulation (mais n'indique pas si l'activite marche ou pas)
    #status=5, probleme que dans l'activite
    #status=4, ok sur tous les parametres avec ou sans activite.
    C0 = (data[:,1] >= 4)

    # The MLE fit has correctly converged (her for seismic parameters only)
    C1 = (data[:,15] > 0.) & (data[:,17] > 0.) & (data[:,16] > 0.) & (data[:,16]!=np.inf) & (data[:,36]>0) & (data[:,13]>0.) 

    # Limite sur Amax
    C2 = (data[:,32] >= 8.0)  & (data[:,32] < 700.0)
    
    # Alpha doit prendre des valeures physiques
    C3 = (data[:,10] < 5.0)

    # Limites sur numax
    if(origin==1):
        C4 = (data[:,14]>2) & (data[:,14]<=100) 
    else:
        C4 = (data[:,14]<250) & (data[:,14] >2.) 

    # Limites sur le SNR
    SNR = np.exp(data[:,12])/data[:,37]
    C5 = (SNR > 1.0) & (SNR < 70)

    # Limites sur Henv et delta_nu
    Dnuinf=0.3	
    Dnusup=0.3
    Hinf=1
    Hsup=3
    if (origin==1):
        Henv_scal =  8.9e6*data[:,14]**(-1.81)	
        Dnu_scal = 0.250*data[:,14]**(0.771)

    else:
        Henv_scal = 1.4e7*data[:,14]**(-1.83)
        Dnu_scal = 0.296*data[:,14]**(0.740)
        
    C6 = ( (data[:,16]-Dnu_scal) / (Dnu_scal*Dnuinf) )**2 * (data[:,16]<Dnu_scal) +\
             ( (data[:,16]-Dnu_scal) / (Dnu_scal*Dnusup) )**2 * (data[:,16]>Dnu_scal) +\
             ((np.exp(data[:,12])-Henv_scal)/(Henv_scal*Hinf))**2 * (np.exp(data[:,12])<Henv_scal) +\
             ((np.exp(data[:,12])-Henv_scal)/(Henv_scal*Hsup))**2 * (np.exp(data[:,12])>Henv_scal) <= 1


    # Restrictions des erreurs relatives trop elevees (> a 15% pour Kepler et 20% pour corot)
    err_Henvp = abs( np.exp(data[:,12] + data[:,13]) - np.exp(data[:,12]) )
    err_Henvn = abs( np.exp(data[:,12]) - np.exp(data[:,12] - data[:,13]) )
    err_pgranp = abs( np.exp(data[:,6] + data[:,7]) - np.exp(data[:,6]) )
    err_pgrann = abs( np.exp(data[:,6]) - np.exp(data[:,6] - data[:,7]) )
    err_taugranp = abs( np.exp(data[:,8] + data[:,9]) - np.exp(data[:,8]) )
    err_taugrann = abs( np.exp(data[:,8]) - np.exp(data[:,8] - data[:,9]) )
    err_tauegranp = abs( np.exp(data[:,25] + data[:,26]) - np.exp(data[:,25]) )
    err_tauegrann = abs( np.exp(data[:,25]) - np.exp(data[:,25] - data[:,26]) )
    err_sigmagranp = abs( np.exp(data[:,38] + data[:,39]) - np.exp(data[:,38]) )
    err_sigmagrann = abs( np.exp(data[:,38]) - np.exp(data[:,38] - data[:,39]) )

 
    C7 = (data[:,15]/data[:,14] < 1.00) & (data[:,17]/data[:,16] < 1.00) & \
             (err_Henvp/np.exp(data[:,12]) < 1.00) & (err_Henvn/np.exp(data[:,12]) < 1.00) & \
             (err_Henvp/np.exp(data[:,12]) > 1e-3) & (err_Henvn/np.exp(data[:,12]) > 1e-3)

    if (origin==1):
        # Le nombre de jour d'observation doit etre > 50 jours afin d'avoir une resolution minimum
        C8 =  (Nbr_jour > 50.) & (Nbr_jour != np.inf) 
    else:
        # On se limite aux longues cadences
        C8 =  (nyquist<300.) 

    
    if (origin==1):
        C9 = np.ones(n,dtype=np.bool)
    else: # limit on Teff
        C9 = (teffvalues < 5700.) & (teffvalues > 4000.)

    # final selection (for seismic parameters)
    good = C0 & C1 & C2 & C3 & C4 & C5 & C6  & C7 & C8 & C9

    # generation of the flags (all kind of indices)
    status = (2**0) * (C0==False) + (2**1) * (C1==False) + (2**2) * (C2==False) + (2**3) * (C3== False) + (2**4) * (C4==False) + (2**5) *(C5==False) + (2**6) * (C6==False)  + (2**7) * (C7==False)  + (2**8) * (C8==False) + (2**9) * (C9==False)

    # selection of the granulation parameters
    status_gran = status.copy()
    good_gran = good.copy()

    Pinf=1	
    Psup=3		
    tauinf=1	
    tausup=3    
    # the fit has properly converged
    C10 = (data[:,7] >0.) & (data[:,9] >0.) & (data[:,11] >0.) & (data[:,26] >0.) & (data[:,39] >0.)
    if(origin==1):
        C11 = np.ones(n,dtype=np.bool)
        
        P_scal = 1.2e7*data[:,14]**(-1.84)	
        tau_scal =2.4e5*data[:,14]**(-0.92)
        sigma_scal = 1.6e7*data[:,14]**(-1.06)
        taue_scal = 7.1e5*data[:,14]**(-1.03)

 
    else:

        # Limitation sur numax
        C11 = (data[:,14] < 100.)
        

        P_scal = 1.4e7*data[:,14]**(-1.94)
        tau_scal =6.5e4*data[:,14]**(-0.57)
        sigma_scal = 2.6e7*data[:,14]**(-1.26) 
        taue_scal = 3.5e5*data[:,14]**(-0.81)
        
    C12 = ((np.exp(data[:,6])-P_scal)/(P_scal*Pinf))**2 * (np.exp(data[:,6])<P_scal) +\
       ((np.exp(data[:,6])-P_scal)/(P_scal*Psup))**2 * (np.exp(data[:,6])>P_scal) +\
       ((np.exp(data[:,8])*1e6-tau_scal)/(tau_scal*tauinf))**2 * (np.exp(data[:,8])*1e6<tau_scal) +\
       ((np.exp(data[:,8])*1e6-tau_scal)/(tau_scal*tausup))**2 * (np.exp(data[:,8])*1e6>tau_scal) <= 1

    # Limite sur alpha
    C13 = (data[:,10] < 5.0) & (data[:,10] > 1.0)

    # Limite sur les erreurs
    C14 = (data[:,15]/data[:,14] < 1.00) & (data[:,17]/data[:,16] < 1.00)   & \
          (data[:,11]/data[:,10] < 1.00) & \
          (err_taugranp/np.exp(data[:,8]) < 1.00) & (err_taugrann/np.exp(data[:,8]) < 1.00)  & \
          (err_tauegranp/np.exp(data[:,8]) < 1.00) & (err_tauegrann/np.exp(data[:,8]) < 1.00) & \
          (err_sigmagranp/np.exp(data[:,38]) < 1.00) & (err_sigmagrann/np.exp(data[:,38]) < 1.00) & \
          (err_pgranp/np.exp(data[:,6]) < 1.00) & (err_pgrann/np.exp(data[:,6]) < 1.00) &  (data[:,11]/data[:,10] > 1e-3)   & \
          (err_taugranp/np.exp(data[:,8]) > 1e-3) & (err_taugrann/np.exp(data[:,8]) > 1e-3)  & \
          (err_tauegranp/np.exp(data[:,8]) > 1e-3) & (err_tauegrann/np.exp(data[:,8]) > 1e-3) & \
          (err_sigmagranp/np.exp(data[:,38]) > 1e-3) & (err_sigmagrann/np.exp(data[:,38]) > 1e-3) & \
          (err_pgranp/np.exp(data[:,6]) > 1e-3) & (err_pgrann/np.exp(data[:,6]) > 1e-3) &\
          (data[:,39] > 1e-3)
        
    good_gran = good_gran & C10 & C11 & C12 & C13 & C14 
    status_gran = status_gran + (2**10) * (C10==False)  + (2**11) * (C11==False)  + (2**12) * (C12==False) + (2**13) * (C13==False) + (2**14) * (C14==False) 
        
        
    # we seach the stars with more than one occurence:
    ID = np.array(data[:,0],dtype=np.int)
    indices = np.setdiff1d(np.arange(ID.size), np.unique(ID, return_index=True)[1]) # indices of targets observed more than once 
    if(len(indices) >0):
            ID2 = ID[indices]
            ID3 = np.unique(ID2)
            print 'total number of stars observed more than once: ' , ID3.size
            sel = np.ones(data.shape[0],dtype=np.bool)
            sel[indices] = False
            # among these stars,  we select the one with the highest frequency resolution
            for i in ID3:
                u = (ID == i)
                j = np.argmin(data[u,47])
                sel[np.where(u)[0][j]] = True

    return  good,good_gran,status,status_gran


def correct_mle(data,origin):
        # ----------------------- Correction des biais ------------------------------ 

    def parabola3(x, d, c, b, a):
	return a*x**3 + b*x**2 + c*x + d

    def parabola4(x, e, d, c, b, a):
        return a*x**4 + b*x**3 + c*x**2 + d*x + e

    def parabola6(x, g, f, e, d, c, b, a):
	return a*x**6 + b*x**5 + c*x**4 + d*x**3 + e*x**2 + f*x + g
    
    if(origin==2): # Kepler data
        coeff_Henv = [ -4.34043521e-01,   1.20314690e-02,  -1.82646966e-05,   4.57993830e-07,  -1.10134464e-08,  5.56520726e-11,  -8.22446538e-14]
        coeff_numax = [  -4.56358871e-03,   2.71633764e-03,  -7.80509407e-05,   8.82377795e-07,  -4.83695849e-09,   1.28056981e-11,   -1.30883292e-14]
        coeff_taueff = [ -6.29564567e-01,  3.73126560e-03,   -1.94779870e-05,  -1.44824896e-08,   3.32078358e-09,  -1.99897245e-11,   3.19547036e-14]
        coeff_sigma2 = [  1.65810917e-02,  1.58826020e-03,   -1.01345563e-04,  1.17988702e-06,   -3.05536484e-09,  -5.88121756e-12,  2.12641452e-14]


        data[:,14] = data[:,14]*(1-parabola6(data[:,14], *coeff_numax))
        data[:,12] = data[:,12] + np.log (1-parabola6(data[:,14], *coeff_Henv))
        data[:,25] = data[:,25] + np.log (1-parabola6(data[:,14] , *coeff_taueff))
        data[:,38] = data[:,38] + np.log (1-parabola6(data[:,14], *coeff_sigma2))
        data[:,39] *=3 # sous-estimation d'un facteur 3
    
    if(origin==1): # CoRoT data
        coeff_Henv   = [ -1.93174955e-01,   1.77090634e-02,  -2.18847834e-04,   7.82959412e-07]
        coeff_numax  = [  1.41421670e-02,   3.79751566e-04,  -1.61576009e-05,   9.30645686e-08]
        coeff_taueff = [ -5.30814119e-01,   5.68263522e-03,  -2.38744719e-05,   6.53339691e-08]
        coeff_sigma2 = [  7.67438116e-03,  -3.49912720e-03,   7.47536484e-05,  -3.34738604e-07]
        data[:,14] = data[:,14]*(1-parabola3(data[:,14], *coeff_numax))
        data[:,12] = data[:,12] + np.log (1-parabola3(data[:,14], *coeff_Henv))
        data[:,25] = data[:,25] + np.log (1-parabola3(data[:,14] , *coeff_taueff))
        data[:,38] = data[:,38] + np.log (1-parabola3(data[:,14], *coeff_sigma2))

        
    return data

fltmin = np.finfo(np.float64).min
fltmax = np.finfo(np.float64).max
lfltmax = math.log(fltmax)
float_bound = lambda  flt :  (fltmax if (flt > fltmax) else  ( fltmin if (flt < fltmin)    else (  0.   if (math.fabs(flt) < 1e-15) else flt )  ) )  
exp_bound = lambda flt : (  lfltmax if (flt > lfltmax) else (-lfltmax if(flt < -lfltmax ) else flt  ) )

def postprocess_one_star(config,StarID,Origin,data,status,status_gran,teffvalue,gen_version):

        record=Record(config)
        record.read_app(StarID,Origin)

        # recording the Teff value
        if(Origin==2):
            if(teffvalue>0.):
                record.Teff = int(round(teffvalue))
            
        # -----------------------------------------------------------             
        # storage
        # -----------------------------------------------------------

        Asym_prec = 0
        Asym_acc = 0
        PrecN = -1.
        Acc = -1.
        AccN = -1.

        # Deltanu from UP [muHz]
        Value = float(data[16])
        Type = 'Deltanu_up'
        Prec = float(data[17])
        Value, Prec , PrecN  = map( float_bound , (Value, Prec , PrecN) )
        Comment = ''
        indice = Indice(config,version=gen_version)
        indice.set_IndiceWithCom (Value, Type, Prec , PrecN, Acc, AccN, Asym_prec , Asym_acc, Comment)
        indice.Status = status
        record.Indices.append(indice)

        # Amax - Peak of the ACF - 2nd pass
        Value = float(data[32])
        Type = 'Amax'
        Prec = -1.
        Value, Prec , PrecN  = map( float_bound , (Value, Prec , PrecN) )
        Comment = ''
        indice = Indice(config,version=gen_version)
        indice.set_IndiceWithCom (Value, Type, Prec , PrecN, Acc, AccN, Asym_prec , Asym_acc, Comment)
        indice.Status = status
        record.Indices.append(indice)

        # numax_up : numax from the UP [muHz]
        Value = float(data[14])
        Type = 'numax_up'
        Prec = float(data[15])
        Comment = ''
        indice = Indice(config,version=gen_version)
        indice.set_IndiceWithCom (Value, Type, Prec , PrecN, Acc, AccN, Asym_prec , Asym_acc, Comment)
        indice.Status = status
        record.Indices.append(indice)

        # H_env : Height of the envelop [ppm^2/muHz]
        lv,le = map(float,(data[12],data[13]))
        lv,le = map(exp_bound , (lv,le))
        Value =  math.exp(lv) 
        Type = 'H_env'
        Prec = math.exp( lv  +  le ) - Value # +sigma
        PrecN = Value - math.exp( lv -  le ) # -sigma
        Value, Prec , PrecN  = map( float_bound , (Value, Prec , PrecN) )
        Comment = ''
        indice = Indice(config,version=gen_version)
        indice.set_IndiceWithCom (Value, Type, Prec, PrecN, Acc, AccN, Asym_prec , Asym_acc, Comment)
        indice.Status = status
        record.Indices.append(indice)


        # alpha_gran : Granulation slope
        Value = float(data[10])
        Type = 'alpha_gran'
        Prec = float(data[11])
        Value, Prec , PrecN  = map( float_bound , (Value, Prec , PrecN) )
        Comment = ''
        indice = Indice(config,version=gen_version)
        indice.set_IndiceWithCom (Value, Type, Prec , PrecN, Acc, AccN, Asym_prec , Asym_acc, Comment)
        indice.Status = status_gran
        record.Indices.append(indice)

        # from now on,  Asym_prec = 1 and PrecN is defined
        Asym_prec = 1

        # P_gran : granulation power [ppm^2/muHz]
        lv,le = map(float,(data[6],data[7]))
        lv,le = map(exp_bound , (lv,le))
        Value =  math.exp(lv) 
        Type = 'P_gran'
        Prec = math.exp( lv  +  le ) - Value # +sigma
        PrecN = Value - math.exp( lv -  le ) # -sigma
        Value, Prec , PrecN  = map( float_bound , (Value, Prec , PrecN) )
        Comment = ''
        indice = Indice(config,version=gen_version)
        indice.set_IndiceWithCom (Value, Type, Prec , PrecN, Acc, AccN, Asym_prec , Asym_acc, Comment)
        indice.Status = status_gran
        record.Indices.append(indice)

        # tau_gran : granulation timescale [sec]
        lv,le = map(float,(data[8],data[9]))
        lv,le = map(exp_bound , (lv,le))
        Value =  math.exp(lv)
        Type = 'tau_gran'
        Prec = math.exp( lv  +  le ) - Value # +sigma
        PrecN = Value - math.exp( lv -  le ) # -sigma
        Value, Prec , PrecN  = map( float_bound , (Value*1e6, Prec*1e6 , PrecN*1e6) )
        Comment = ''
        indice = Indice(config,version=gen_version)
        indice.set_IndiceWithCom (Value, Type, Prec , PrecN, Acc, AccN, Asym_prec , Asym_acc, Comment)
        indice.Status = status_gran
        record.Indices.append(indice)

        # taue_gran: e-folding time of the granulation [sec]
        Value = math.exp(float(data[25]))*1e6
        lv,le = map(float,(data[25],data[26]))
        lv,le = map(exp_bound , (lv,le))
        Value =  math.exp(lv)
        Type = 'taue_gran'
        Prec = math.exp( lv  +  le ) - Value # +sigma
        PrecN = Value - math.exp( lv -  le ) # -sigma
        Value, Prec , PrecN  = map( float_bound , (Value*1e6, Prec*1e6 , PrecN*1e6) )
        Comment = ''
        indice = Indice(config,version=gen_version)
        indice.set_IndiceWithCom (Value, Type, Prec , PrecN, Acc, AccN, Asym_prec , Asym_acc, Comment)
        indice.Status = status_gran
        record.Indices.append(indice)

        # sigma2_gran: Mean-square of the granulation brigthness fluctuations [ppm^2]
        lv,le = map(float,(data[38],data[39]))
        lv,le = map(exp_bound , (lv,le))
        Value =  math.exp(lv) 
        Type = 'sigma2_gran'
        Prec = math.exp( lv  +  le ) - Value # +sigma
        PrecN = Value - math.exp( lv -  le ) # -sigma
        Value, Prec , PrecN  = map( float_bound , (Value, Prec , PrecN) )
        Comment = ''
        indice = Indice(config,version=gen_version)
        indice.set_IndiceWithCom (Value, Type, Prec , PrecN, Acc, AccN, Asym_prec , Asym_acc, Comment)
        indice.Status = status_gran
        record.Indices.append(indice)


        # write the indices
        record.write(StarID, Origin)

        return 1

def main():
    def usage():
        print "usage : postprocess [options] name.xml origin file_teff"
        print "origin: origin of the input data (1 for Corot, 2 for Kepler)"
        print "file_teff: file providing the Teff, for Kepler target only"
        print "Options:"
        print "-p : prompt the user (not compatible with multiprocessing)"
        print "-l <file> : process only the stars with the IDs listed in <file>"
        print "-M <n> : do multiprocessing with <n> processors (this implies the quiet mode)"
	print "-X <file> : exclude from the selection the star ID in <file>"
        print "-t : test mode, do not save the results"
        print "-v : return program version"

    if(len(sys.argv)<2):
        usage()
        sys.exit(2)
    try:
        opts,args = getopt.getopt(sys.argv[1:],"hptl:M:vX:")

    except getopt.GetoptError, err:
        print str(err)
        usage()
        sys.exit(2)

    nproc = 1
    prompt = False
    test = False
    flist = ''
    Xlist = ''
    for o, a in opts:
        if o == "-h" :
            usage()
            sys.exit(1)
        elif o == "-p":
            prompt = True
        elif o == "-M":
            nproc = int(a)
        elif o == "-t":
            test = True
        elif o == "-l":
            flist = a
        elif o == "-X":
            Xlist = a
        elif o == "-v":
            print __version__
            sys.exit(1)
        else:
            print "unhandled option %s" % (o)
            sys.exit(1)

    nargs = len(args)

    if nargs < 2 :
        print "missing arguments"
        usage()
        sys.exit()


    if (nproc > 1):
        prompt = False

    config=args[0]
    Origin=int(args[1])

    if( (Origin !=2) & (Origin !=1)):
        print 'Only Kepler or CoRoT data can be processed'
        sys.exit(1)
        
    if ( ((nargs > 2) and Origin==1) or ((nargs > 3) and Origin==2) ):
        print "too many arguments"
        usage()
        sys.exit()
  

    if(Origin==2):
        if nargs < 3 :
            print "missing arguments"
            usage()
            sys.exit()
            
        file_teff = args[2]
        data_teff = np.genfromtxt(file_teff,delimiter=',',usecols=(1,3))
    else:
        file_teff = None
        data_teff = None

    output = commands.getstatusoutput('mle_up -v')
    gen_version = re.sub("\n$","",output[1])
    print "Version:",__version__
    print "Version of the indice generation program:" , gen_version

    #lecture des fichiers binaires des etoiles a etudier
    doc = minidom.parse(args[0])
    elem= doc.getElementsByTagName('pp_output')
    pp_output= elem.item(0).firstChild.nodeValue
    StarList=GF.StarID_List(pp_output)
##    ge_version = os.system("process -v")
    N_object = len(StarList)	#Number of object that we want to process

    if (N_object == 0):
            print 'No stars ! nothing to do !'
            sys.exit(1)

    if (flist != ''):
    # The ID of the stars we want to process are taken from the file <flist> 
        ID = np.loadtxt(flist)
        NewStarList = []
        for i in range(N_object):
            k = np.where( ID == StarList[i][0])[0]
            if ( len(k) > 0 ):
                NewStarList.append(StarList[i])
        n = len(NewStarList)
        if ( n > 0):
            print ('Of the %d stars listed in %s, %d are available') % (ID.size,flist,n)
            StarList = NewStarList
            if (prompt and (nproc < 2) ):
                s = raw_input("Must we process these stars,  yes (y) or not (n) ?\n")
                if (s.lower() == 'n'  ):
                    sys.exit( 0)
        else:
            print ('None of the stars listed in  %s are available') % (flist)
            sys.exit(1)

    N_object = len(StarList)	#Number of object that we want to analyse

    print "Number of object to analyse:",N_object
    
    # read the data
    fdata = re.sub('\.xml','_mle_UP.npy',config)
    data0 = np.load(fdata)           	

    # extract the data correspondong to our list of stars
    ID0 = np.array(data0[:,0],dtype=np.int)
    data = np.zeros((N_object,data0.shape[1]))
    StarID = np.zeros(N_object,dtype=np.int)
    i = 0
    count = 0
    for Star in StarList:
        Origini = Star[1]
#        print StarID[i],Origini,Origin,( Origini == Origin ),Origini-Origin
        if( Origini == Origin ):
            select = (ID0 == Star[0])
            n = select.sum()
            if (n >= 1) :
                data[count,:] = (data0[select,:]) [0,:]
                StarID[count] = Star[0]
                count += 1
            if (n>1): # more than on instance
                    print "Warning ! More than one instance of star ID =",StarID[i]
        i += 1
    if(count <1):
        print "No data associated with our list of stars, check  the origin of the stars or the coherence of the data"
        sys.exit(1)


    N_object = count
    data = data[0:N_object,:]
    StarID = StarID[0:N_object]
    print ('Nb of stars taken from %s : %i') % (fdata, N_object )

    name = re.sub('\.xml','',config)
    # exclude the stars listed in Xlist
    if( Xlist != ''):
	excluded = np.loadtxt(Xlist,dtype=np.int)
	selected = (np.in1d(StarID,excluded) == False)
	print ('Nb of stars discarded according to %s : %i') % ( Xlist , (selected==False).sum())
	data = data[selected,:]
	StarID = StarID[selected]
	N_object = StarID.size
    print 'Nb of stars for which indices must be inserted into the DB:', N_object

    # cross-matching with the Teff data (Kepler targets only)
    if(Origin==2):
        IDTeff = np.array(data_teff[:,0],dtype=np.int)
        match = np.in1d(StarID,IDTeff)
        nmatch = match.sum()
        print 'Number of Kepler targets with Teff values: ', nmatch
        teffvalues = np.zeros(N_object)
        adr = np.where(match)[0]
        for i in adr:
            ID = StarID[i]
            j = np.argmin(np.abs(IDTeff-ID))
            if( math.fabs(IDTeff[j]-ID) < 1e-10) :
                teffvalues[i] = data_teff[j,1]
            
    else:
        teffvalues = -np.ones(N_object) # Teff not available, the value -1 is assigned
        
    # correct for bias
    data = correct_mle(data,Origin)
    
    # select the "good" data
    good,good_gran,status,status_gran =  select_mle(name,data,Origin,teffvalues)

    print ("%d valid indices") % (status == 0).sum()
    print ("%d valid granulation parameters") % (status_gran == 0).sum()

    print 'Nb of selected objects:',good.sum()

    np.savetxt(name+'.all',data[:,0],fmt='%d')
    np.savetxt(name+'.selected',data[good,0],fmt='%d')
    np.savetxt(name+'.granselected',data[good_gran,0],fmt='%d')
    np.save(name+'_mle_UP_COR.npy',data) # save the corrected data

    if(test):
        sys.exit(0)
    

    print 'Storing the indices'
    count = 0
    print('start :{}'.format(datetime.datetime.now()))

    # multi-processing
    if (nproc>1):
        nproc = min(nproc,N_object)

        # define the handler function
        def term_handler(signum, frame):
            print "TERM/INT signal received. Exiting this job."
            sys.exit(-1)
        signal.signal(signal.SIGTERM, lambda signum, stack_frame: sys.exit(1))

        def init_worker():
            signal.signal(signal.SIGINT, term_handler)
            signal.signal(signal.SIGTERM, term_handler)

        pool = multiprocessing.Pool(nproc,init_worker)

        outputs =  [pool.apply_async(postprocess_one_star,(config,StarID[i],Origin,data[i,:],status[i],status_gran[i],teffvalues[i],gen_version)) for i in range(N_object)]

        pool.close()

        def finish(signum,frame):
            print "TERM/INT signal received. Exiting. The results are backup"
            pool.terminate()
            pool.join()
            # collecting the results
            i = 0
            for output in outputs:
                    if (output.ready()):
                        i = i + 1
            sys.exit(-1)

        signal.signal(signal.SIGTERM,finish)
        signal.signal(signal.SIGINT,finish)

        # waiting for the completion
        countp = 0
        while ( count < N_object):
            count = 0 
            for output in outputs:
                    if (output.ready()):
			if(output.get()==1):
                        	count = count + 1
            time.sleep(1)
            if( count > countp +  float(N_object)*1e-2 ):
                    print ("%5.0f %% completed") % (float(count)/float(N_object)*100.)
                    sys.stdout.flush()
                    countp = count
        pool.join()

    # mono-processing
    else: 

        stop = False
         
        for i in range(0,N_object):
            print i+1,'/',N_object
            try:
                    result = postprocess_one_star(config,StarID[i],Origin,data[i,:],status[i],status_gran[i],teffvalues[i],gen_version)
		    if(result==1):
			count += 1
            except KeyboardInterrupt:
                print 'Keyboard interrupt'
                sys.exit(1)
            except:
                traceback.print_exc()
                print ('some errors occur while processing star ID=%d') % (StarID[i])

    print('end :{}'.format(datetime.datetime.now()))      
    print ("Indices for %d stars among %d were correctly stored") % (count,N_object)




if __name__ == "__main__":
    main()

