#!/usr/bin/env python
'''
	Implementation of the MLE-UP method

	It is part of the Stellar Seimic Indices pipeline

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

	Copyright (C) 2016 by R. Peralta and R. Samadi
'''

import getopt
import numpy as np
import math
import scipy
import os,sys, traceback,signal
import re
from xml.dom import minidom
import datetime
import multiprocessing

from mpl_toolkits.mplot3d import Axes3D
from matplotlib.ticker import LinearLocator, FormatStrFormatter
import matplotlib.pyplot as plt
from Struc.Record import Record
from Struc.Indice import Indice
import GestionFiles.GF as GF
from scipy.stats import chi2

from background_models import *
import Autocorrelation as auto
import ssilib 
from hessian import *
import universal_pattern as UP
from smooth_func import *
from mapping import *
import time
# import sls
 

__version__ = "1.3"

# 1.3 : New parameters ACFThreshold, stars with ACF < ACFThreshold are skipped
# 1.2 : Initial guess value alpha_gran = 2 
# 1.1 : without folding
# 1.0 : initial version


# ModelType = 1 : pseudo-Lorentzian function for the granulation component
# ModelType = 2 : Exponential function for the granulation component
ModelType = 1
agran = True	# For pseudo-Lorentzian function (ModelType==1) : agran==True => slope of granulation free; agran==False => slope of granulation fixed

""""""""""""""""""""""""""""""" BACKGROUND FIT """""""""""""""""""""""""""""""

def background_mle_fit (f, PSD, numax, delta_nu_ACF, nu0 , nu1, origin, StarID, nyquist, activity=False, plot=True, verbose=True, refit=False, maxiter = 200):

	if (nu1 == 'nyquist'): nu1 = nyquist

        cdtn = 0. 	# Hessian matrix condition number
        status = 0 	# status of fit convergence
	recompute = False
        
	f = f*1e6		# In order to obtain the frequency in [muHz]
	fprime = f

	sampling = 0.5/f[-1]*1e6
	N = len(PSD)
	acf = np.fft.irfft(np.append(0.,PSD *  (f < nu1*1.5)  ),n=N*2)	
        acf = acf[1:N]/acf[0] 
        time = (np.arange(N-1)+1.)*sampling
        eft = np.interp(math.exp(-1),acf[::-1],time[::-1])
        tau_act = eft
        tau_gran = eft /10.
        numax_gran=  10.*(tau_gran/1e5)**(-1./0.89)
        if (verbose): print 'e-folding time, numax :',eft , numax_gran

	y = np.roll(f,1)
	dif = f-y   
	df = np.median(dif)		# Frequency resolution
	Nbr_jour = 1/df * 1e6 * 1/86400.
	if verbose: print ("Spectrum frequency resolution: %f [muHz], corresponding to around %i days of observations") % (df, int(Nbr_jour))
	if verbose: print ("Nyquist frequency : %f [muHz]") % (nyquist)


	""""""""""""""""""""""""""""""" AREA SELECTION """""""""""""""""""""""""""""""

	#if(plot):
        #        plt.figure(1)
        #        plt.clf()
        #        plt.plot(f,PSD,'r')	#, label=("raw PSD")
	#	 plt.legend(loc=1)


	if (origin == 1):		# Area selection only for CoRoT stars
   		f_day = 11.57		# Day frequency in [muHz] (= 1/86400[sec])
    		f_orb = (1./6184)*1e6	# The orbital period is 6184 s

    		N_freq = 30		# we want to delete alias frequencies:
    		f_start= 1		# from f_day*2 => 23.14 muHz to f_day*15 => 173.55 muHz

    		for i in range(f_start,N_freq):	
        		freq = f_day*i
        		if i<=12:
            			eps = 0.50
        		else:
            			eps = 1.5

        		# Removing alias from data
        		index = np.where( (f>=(freq-eps)) & (f<=(freq+eps)) ) [0]
        		PSD = np.delete(PSD,index)
        		f = np.delete(f,index)

	#if(plot):
        #        plt.figure(1)
	#	plt.xlabel('Frequency [muHz]')
	#	plt.ylabel('Power [ppm2/muHz]')
	#	plt.title( ('Star ID: %i') % (StarID) )
	#	plt.plot(f,PSD,'b')	#, label=("PSD ss parasites")


   	""""""""""""""""""""""""""""""" BACKGROUND PARAMETERS """""""""""""""""""""""""""""""

 	# White noise
        #ind = np.where( (f>=min(5.*numax,np.max(f))-3*delta_nu) & (f<=5.*numax+3*delta_nu))[0]
	ind = np.where( (f>=min(3.*numax,np.max(f)-20.) ) )[0]
        W = np.median(PSD[ind])	# median of 6*deltanu around 3*numax

        # Harvey-like profil parameters - granulation, see Mathur et al. (2011 ApJ 741 119)
        H_gran = [ 5e5*(numax/10.)**-1.90, 1e5*(numax/10.)**-0.89 * 1e-6, 3.]
        ind = np.where( (f>=max(numax/25.,f[1])) & (f<=max(numax/10.,f[1])))[0]
        H_gran[0] = np.median(PSD[ind])

	if (ModelType == 1) & (agran==True) : # pseudo-lorentzian function and free granulation slope
		H_gran[2] = 2.
		"""
		if( numax < 20.):
			H_gran[2] = 4.
		elif (numax < 200.):
			H_gran[2] = (1.5-4.)*(numax-20.)/(200.-20.)+ 4.
		else:
			H_gran[2] = 1.5
		"""

	elif (ModelType == 1) & (agran==False) : # pseudo-lorentzian function and free granulation slope
		H_gran[2] = 3. 

	elif (ModelType == 2): # Exponential function
		H_gran[2] = 1. 
		H_gran[0] /= math.log(2)

	else:
		print ("Unhandled model type = %i ")  % (ModelType)
		sys.exit(2)


        # Harvey-like profil parameters - activity, cf Hekker et al.
	H_act = [ 1e-10 , 10.* H_gran[1], 2.]
        if (activity):
    	        ind = np.where( (f<=max(numax/100.,f[1])) )[0]
    	        H_act[0] = np.median(PSD[ind])

	# Universal pattern parameters - Oscillation spectrum, see Mosser et al. (2011 A&A 525 L9)
	ampli = 1e8*(numax**(-2.38))
	G_env = [delta_nu_ACF, numax, ampli]		

	# Guess parameters
        p_guess = np.zeros(10)
	p_guess[0] = W
	p_guess[1:4] = H_gran
	p_guess[4:7] = G_env
	p_guess[7:10] = H_act

	env_width = 0.66*(numax**0.88)
	norders	 = int(round(env_width/delta_nu_ACF)*2)
	norders = norders + ( (norders %2) ==0)		 # to have an odd number
	if(verbose):
		print ('envelope width=%f norders=%i') % (env_width,norders)


	""""""""""""""""""""""""""""""""""""" 1st step """""""""""""""""""""""""""""""""""""
	""" Granulation free ; Activity free ; Envelope fixed """
	""" Guess parameters derived from the Autocorrelation function """

	if(verbose):
		print 
		print '--- 1st step: fitting Gran + Act(fixed slope) alone'

	# Initialisation des compteurs pour les statistiques
	compt_avAct = 0
	compt_ssAct = 0
	compt_uncer = 0
	compt_refit = 0

	step2_numax_Dnu = 0
	step2_Dnu_numax = 0		
	step2_step1 = 0	


	# setting the free parameters
	free = np.ones(10, dtype=np.bool)
	free[0] = True		# White noise: free
	free[1:4] = True	# Granulation: free
	if (ModelType == 2) | ((ModelType == 1) & (agran==False)) : 
		free[3] = False # slope fixed		
	free[4:7] = False 	#    Envelope: fixed
	free[7:10] = activity	#    Activity: depends on the configuration (.xml)
	free[9] = False 	# The slope of the activity is fixed

	# some parameters are converted to log value
	p_guess = BGModel_log2lin(p_guess,inv=True) 	# inital guess values
	p00 = p_guess.copy()				# copy of guess values
	p = p_guess.copy()				# final values
        perror = np.zeros(p_guess.size)			# error


	if(verbose):	
		print 'guess parameters:', BGModel_log2lin(p_guess,inv=False)


        # Fit area : if the activity is off, we need to avoid low frequencies for a good granulation fit. Otherwise, we release the granulation slope
	if (activity==False): 
		ind=np.where( f >= numax/20. ) [0]	# & (f<= min(4.*numax,nu1*2.0)) )[0]		
		if(verbose): print ("Activity is off: removing low frequency below numax/20 (=%f)") % (numax/20.)

	else: 
		ind=np.where(f>=0.)[0]
		Nbr_param_avAct = len(np.where(free==True)[0])


        if(verbose):
		print 
		if (activity==False): print 'fitting without the activity component...'
		else: print 'fitting with the activity component...'


	try:
	    (popt,likelihood,direct,iter,funcalls,warnflag) = scipy.optimize.fmin_powell(BGModel_likelihood_powell, p_guess[free], args=(f[ind],PSD[ind],free,p_guess,nyquist,norders,ModelType) , full_output = True , disp=verbose, maxiter = maxiter, xtol=1e-5, ftol=1e-5)
	except:
	    warnflag = 3	# Problem in adjusting


	if (warnflag < 1): 

		if(verbose):
			p_bis = p_guess.copy()
			p_bis[free] = popt
			print 'Results:', BGModel_log2lin(p_bis,inv=False)

		if (activity==False): 
			p[free] = popt		# We keep the optimal parameters from the fit without activity component
			status = 1
			if (p[3] > 5.0) :	# If the slope of granulation is too high, we recompute with an other Pgran guess and on all the spectrum

				if(verbose): 
					print 
					print 'the slope of granulation is too high: re-computation on all the spectrum'

				ind=np.where(f>=0.)[0]
				p_guess[1] = math.log( np.median( np.where( (f>=max(numax/75.,f[1])) & (f<=max(numax/15.,f[1])))[0] ))	# New guess for P_gran
        			
				(popt,likelihood,direct,iter,funcalls,warnflag) = scipy.optimize.fmin_powell(BGModel_likelihood_powell, p_guess[free], args=(f[ind],PSD[ind],free,p_guess,nyquist,norders,ModelType) , full_output = True , disp=verbose, maxiter = maxiter, xtol=1e-5, ftol=1e-5)
	
				if (warnflag < 1): 
					status = 1
					p[free] = popt
					if(verbose): print 'Results:', BGModel_log2lin(p,inv=False)

				else:
					status = 0
					if(verbose): print 'convergence failed'
			
	else:
		if(verbose): print 'convergence failed'


	# If Activity is on, we test if the fit is better without this component
	if (activity==True) & (warnflag < 1):

        	if(verbose):
			print 
			print 'fitting without activity...'

		free[7:10] = False		# Turn off the Activity
		p_ssAct = p_guess.copy()
		ind2 = np.where( (f>=max(numax/75.,f[1])) & (f<=max(numax/15.,f[1])))[0]
		p_ssAct[1] = math.log(np.median(PSD[ind2]))
		p_ssAct[7] = 1e-10			# P_act = 0
		Nbr_param_ssAct = len(np.where(free==True)[0])


		try:
		   (popt_ssAct,l_ssAct,direct,iter,funcalls,warnflag) = scipy.optimize.fmin_powell(BGModel_likelihood_powell, p_ssAct[free], args=(f[ind],PSD[ind],free,p_ssAct,nyquist,norders,ModelType) , full_output = True , disp=verbose, maxiter = maxiter, xtol=1e-5, ftol=1e-5)
		except: 
		   warnflag = 3	# Problem in adjusting


		if (warnflag < 1):
	
			if(verbose):
			   p_bis2 = p_ssAct.copy()
			   p_bis2[free] = popt_ssAct
			   print 'Results:', BGModel_log2lin(p_bis2,inv=False)

			log_ratio = likelihood - l_ssAct		# ratio = l_(p+q) - l_(p) 
			X = -2*(log_ratio)				# X: random variable following a chi2 statistic with q degree(s) of freedom
			Px = 0.99					# Px: Confidence level
			dof = Nbr_param_avAct - Nbr_param_ssAct		# Number of degree of freedom (dof) = q
			threshold = chi2.ppf(Px,dof)			# Threshold: depends on the confidence level (Px) and the dof
			# X >= +threshold : they are some activity with the probability Px ;
			# X <= -threshold : they are no activity with the probability Px ;
			# -threshold <= X <= +threshold : uncertain area => we remove the activity but we skip very low frequencies

			if(verbose):
				print
				print ('likelihood value with activity = %f') % (likelihood)
				print ('likelihood value without activity = %f') % (l_ssAct)
				print ('X = %f, dof = %i, threshold = %f at %f percent') % (X, dof, threshold, Px)
				print 

			# Traitement des differents cas de figure de la comparaison avec et sans activite
			if (ModelType == 2) | ((ModelType == 1) & (agran==False)) : popt_ssAct = np.array([popt_ssAct[0],popt_ssAct[1],popt_ssAct[2],p_guess[3]])	#adding the value of agran in order to have the correct size of popt_ssAct

			if (X >= threshold) & (popt[1] > popt[0]) & \
			   ( ( (ModelType == 1) & (popt[4] > popt[1]) & (popt[3] < 5.0) & (popt[3] > 1.0) ) | \
			     ( (ModelType == 2) & (popt[3] > popt[1]) ) ):

				if(verbose): print ' => We keep the background activity component'
				free[7:10] = True		# Activity on
				free[9] = False 
				p[free] = popt			# We keep the optimal parameters from the fit with activity
				status = 1
				compt_avAct = 1
				

			elif (X<=-threshold) & (popt_ssAct[1]<10*(5e5*(numax/10.)**-1.90)) & (popt_ssAct[1]>popt_ssAct[0]) & \
		  		( ( (ModelType == 1) & (popt_ssAct[3]<5.0) & (popt_ssAct[3]>1.0) ) | (ModelType == 2) ) : 
			# P_gran proportionnel a 5e5*(numax/10.)**-1.90 d'apres Mathur et al. 2011. 

				if(verbose): print ' => We remove the background activity component'
				activity=False			# Activity off 
				p[free] = popt_ssAct		# We keep the optimal parameters from the fit without activity
    	        		p[7] = 1e-10			# P_act = 0
				status = 1
				likelihood = l_ssAct
				compt_ssAct = 1


			else :
				if(verbose): 
					print ' => Uncertain area: We remove the activity component skipping low frequencies'
					print ("Fitting without activity and skipping low frequencies, below numax/20. (=%f)") % (numax/20.)
					print 

				compt_uncer = 1

				activity=False				# Activity off 
				ind = np.where( f >= numax/20. ) [0]	# We skip very low frequencies 


	       			(popt_cut,likelihood,direct,iter,funcalls,warnflag) = scipy.optimize.fmin_powell(BGModel_likelihood_powell, p_ssAct[free], args=(f[ind],PSD[ind],free,p_ssAct,nyquist,norders,ModelType) , full_output = True , disp=verbose, maxiter = maxiter, xtol=1e-5, ftol=1e-5)
				

				if (warnflag < 1): 
					p[free] = popt_cut
	    	        		p[7] = 1e-10			# P_act = 0
					status = 1
					if(verbose): 
						print 'Results:', BGModel_log2lin(p,inv=False)
						print 
				
				else: 
					if(verbose): print 'convergence failed'
					status = 0


		else:
			if(verbose): print 'convergence without activity failed'
			#free[7:10] = True			# Activity on
			#free[9] = False 
			#p[free] = popt				# We keep the optimal parameters from the fit with activity
			#if(verbose): print 'optimal parameters of the 1st fit without activity:', BGModel_log2lin(p,inv=False)
			#ind = np.where(f>=0.) [0]		# We keep all the spectrum frequencies 



	if(status == 1):	#Convergence ok

		# Computation of the likelihood with background + oscillations parameters from the fit (+ scaling law)
		if(ModelType ==1):
			if (p[3]!=2):	
				taue = ssilib.eft(math.exp(p[2]),p[3],ssilib.plf)*1e6	# Computation of tau_eff (taue) 
			else:		
				taue = math.fabs(math.exp(p[2]))*1e6		# if alpha==2 (Lorentzian function), taue = tau
		elif(ModelType ==2):
				taue = ssilib.eft(math.exp(p[2]),p[3],ssilib.pef)*1e6

		numax_gran = 10.*(taue/1e5)**(-1./0.89)				# we obtain a new guess for numax from the granulation time-scale
		p0 = p.copy()
		p0[5] = numax_gran
		p0[4] = 0.280 * (numax_gran)**0.747
		l_gran  = BGModel_likelihood(p0,f[ind],PSD[ind],nyquist,norders,ModelType,sigma2var=False)	# likelihood value given with oscillations from the scaling law

		if(verbose):	
			print 
			print ('Oscillation parameters from ACF: numax = %f, deltanu = %f, likelihood = %f') % (numax, delta_nu_ACF, likelihood)
			print ('Oscillation parameters derived from the granulation-numax scaling law: numax = %f, deltanu = %f, likelihood = %f') % (numax_gran, p0[4], l_gran)
			print 


		# Determination of the best guess for deltanu depending on the best likelihood given by the granulation and ACF
		if (l_gran <= likelihood) & (numax_gran > nu0*0.8) & (numax_gran < nu1*1.3) & (p[1] > p[0]) & (p[1] != p[7]) : #& (p0[3] < 10.0) :
		   p = p0		#if gran+act (or only granulation) give better oscillation parameters than ACF, we only keep its values
		   delta_nu = p[4]	# All parameters derived from the granulation fit
		   numax = numax_gran
		   likelihood = l_gran
		   if(verbose): print " => We keep the oscillation parameters derived from the granulation-numax scaling law"


		elif (l_gran > likelihood) & (p[1] > p[0]) & (p[1] != p[7]) : #& (p[3] < 10.0):
		   delta_nu = delta_nu_ACF	# Oscillation parameters derived from the ACF
		   if(verbose): print " => We keep the oscillation parameters given by the ACF"


		else:		
		   	delta_nu = delta_nu_ACF
		   	if (activity==False) : p[7] = 1e-10	# we remove the activity component
	       	   	if(verbose): print "Problem in the results of the fit => we keep oscillations parameters from the ACF"


	       	p[6] = math.log(1e8*(numax**(-2.38)))

		if(verbose):
		      print 
		      print '1st step - Optimal parameters:', BGModel_log2lin(p,inv=False)
		      print ('New numax value = %f, deltanu = %f, Henv = %f') % (numax,delta_nu, p[6])



	if(plot):
		plt.figure(2)
		plt.clf()
		plt.xlabel('Frequency [muHz]')
		plt.ylabel('Power [ppm2/muHz]')
		plt.title( ('Star ID: %i') % (StarID) )
		plt.plot(f,PSD,'r', label=("PSD"))
		plt.plot(f,BGModel(f,p_guess,nyquist,norders,ModelType),'c', label=("guess"))
		if(status==1):
			plt.plot(f,BGModel(f,p,nyquist,norders,ModelType),'g', label=("step 1"))
		plt.loglog()
		plt.legend(loc=1)





	""""""""""""""""""""""""""""""""""""" 2nd step """""""""""""""""""""""""""""""""""""
	""" likelihood mapping as a function of deltanu and numax """
	""" delta_nu guess derived from either the granulation scaling law numax or the ACF numax """

	if(status == 1):

		if(verbose):
			print 
			print 	
			print '--- 2nd step: l mapping in function of numax then deltanu'

		
		#------------------------- Choix mapping suivant likelihood
		# Compute the likelihood of the first step
		degrad_spec = False	# The spectrum is smoothed to mitigate mixed modes effects

		nbin = max( (int(math.floor(p[4]/100./df)) , 1) )
		res = nbin*df  # resolution
		N = int(math.floor(len(PSD)/float(nbin)))

		if ( (nbin > 1) & (degrad_spec) ):
			if (verbose): print 'summing the PSD over ',nbin,' bins'
			PSD_map = ssilib.rebin1d(PSD[0:N*nbin],N)/nbin
			f_map = ssilib.rebin1d(f[0:N*nbin],N)/nbin
		else:
			PSD_map = PSD
			f_map = f


		if ( (nbin > 1) & (degrad_spec) ):
			l_step1 = BGModel_likelihood(p, f_map, PSD_map, nyquist, norders,ModelType, sigma2var=False)
			if (verbose): print 'Likelihood of the first step (binned spectrum):', l_step1
		else :
			l_step1 = BGModel_likelihood(p, f_map, PSD_map, nyquist, norders,ModelType, sigma2var=False)
			if (verbose): print ('Likelihood of the first step: %f') % (l_step1)


		# Mapping autour de numax
		if(verbose):
			print 
			print '   Mapping of numax puis delta_nu'

		p_map = p.copy()

		numax1, likelihood1 = optimal_numax(StarID, f_map, PSD_map, p_map, nyquist, norders,ModelType, degrad_spec=False, verbose=verbose, plot=False, win=10)
		p_map[5] = numax1
		p_map[4] = 0.274 * numax1**0.757
		p_map[6] = math.log(4e7*(numax1**(-2.38)))
		if(verbose): print ('deltanu guess = %f') % (p_map[4])

		delta_nu1, likelihood1 = optimal_dnu(StarID, f_map, PSD_map, p_map, nyquist, norders,ModelType, degrad_spec=False, verbose=verbose, plot=False, win=12)

		if(verbose): print ('Results: numax1 = %f, delta_nu1 = %f, likelihood1 = %f') % (numax1, delta_nu1, likelihood1)


		#----------------------------------------------
		if(verbose):
			print 
			print '   Mapping of delta_nu puis numax'

		p_map = p.copy()

		delta_nu2, likelihood2 = optimal_dnu(StarID, f_map, PSD_map, p_map, nyquist, norders,ModelType, degrad_spec=False, verbose=verbose, plot=False, win=14)
		p_map[4] = delta_nu2
		p_map[5] = (math.fabs(delta_nu2)/0.274)**(1/0.757)
		p_map[6] = math.log(4e7*(p_map[5]**(-2.38))) 
		if(verbose): print ('numax guess = %f') % (p_map[5])

		numax2, likelihood2 = optimal_numax(StarID, f_map, PSD_map, p_map, nyquist, norders,ModelType, degrad_spec=False, verbose=verbose, plot=False, win=16)

		if(verbose): print ('Results: numax2 = %f, delta_nu2 = %f, likelihood2 = %f') % (numax2, delta_nu2, likelihood2)


		if (likelihood1 <= likelihood2) & (likelihood1 <= l_step1):
			p[4] = delta_nu1
			p[5] = numax1
			p[6] = math.log(4e7*(numax1**(-2.38)))

			if(verbose):
				print 
				print '   => We keep the results of the 1st mapping:'
      				print ('New numax = %f, deltanu = %f and likelihood = %f') % (numax1,delta_nu1,likelihood1)

			step2_numax_Dnu = 1

		elif (likelihood2 < likelihood1) & (likelihood2 < l_step1):
			p[4] = delta_nu2
			p[5] = numax2
			p[6] = math.log(4e7*(numax2**(-2.38)))

			if(verbose): 
				print 
				print '   => We keep the results of the 2sd mapping:'
	      			print ('New numax = %f, deltanu = %f and likelihood = %f') % (numax2,delta_nu2,likelihood2)

			step2_Dnu_numax = 1

		else:
			if(verbose): 
				print 
				print '   => We keep the results of the 1st step:'
				print ('numax = %f, deltanu = %f and likelihood = %f') % (numax,delta_nu,l_step1)

			step2_step1 = 1		



	""""""""""""""""""""""""""""""""""""" 3rd step """""""""""""""""""""""""""""""""""""
	""" Oscillations free """
	""" envelope derived from the scaling law """


	if(status == 1):

		if(verbose):
			print 
			print 	
			print '--- 3rd step: fitting the background with free envelope (Dnu+numax+Aenv)'

		#numax = (math.fabs(delta_nu)/0.274)**(1/0.757) 		# ref. M13		
		free[0:4] = False	# Granulation: fixed
		free[4:7] = True	# Oscillations: free
		free[7:10] = False	# Activite: fixed
		#free[6] = False	# ampli

		p0[:] = p[:]
		ind_osc = ( f >= max(numax-6*delta_nu,f[0]) ) & ( f <= min(numax+6*delta_nu,f[-1]) )


		if(verbose):	
			print 'fitting...'

		(popt,likelihood,direct,iter,funcalls,warnflag) = scipy.optimize.fmin_powell(BGModel_likelihood_powell, p0[free], args=(f[ind_osc],PSD[ind_osc],free,p0,nyquist,norders,ModelType) , full_output = True , disp=verbose, maxiter = maxiter, xtol=1e-5, ftol=1e-5)


		if (warnflag < 1):
			status = 2
			p[free] = popt
			p[4:6] = np.abs(p[4:6])
			delta_nu = p[4]
			numax = p[5]
			#p[6] = math.log(4e7*(numax**(-2.38)))

			if(plot):
		                plt.figure(2)
				plt.plot(f,BGModel(f,p,nyquist,norders,ModelType),'b', label=("step 3"))
				plt.legend(loc=1)

			if(verbose):
				print 
				print '3rd step - Optimal parameters :', BGModel_log2lin(p,inv=False)
				print ('New numax value = %f, deltanu = %f , Henv = %f and likelihood = %f') % (numax,delta_nu,p[6],likelihood)

		else:
			if(verbose): 
				print 'convergence failed'

	
	
	""""""""""""""""""""""""""""""""""""" 4th step """""""""""""""""""""""""""""""""""""
	""" All parameters free except the slope of the activity """

	if(status == 2):

		if(verbose):
			print 
			print 
			print '--- 4th step: fitting all parameters except the slope of the activity'

		if (activity==True) & (p[7]<=1e-10) :	# if activity is on but Pact is null, so we remove the activity component of the last fit
			activity=False
			ind = np.where( f >= numax/20. ) [0]

		free[0:4] = True	# Granulation: free
		if (ModelType == 2) | ((ModelType == 1) & (agran==False)) : 
			free[3] = False # slope fixed	
		free[4:7] = True	#    Envelope: free
		free[7:10] = activity	#    Activite: config
		free[9] = False	 	# The slope of the activity is fixed
		p0[:] = p[:]

		if(verbose):
			print 'fitting...'

		(popt,likelihood,direct,iter,funcalls,warnflag) = scipy.optimize.fmin_powell(BGModel_likelihood_powell, p0[free], args= (f[ind],PSD[ind],free,p0,nyquist,norders,ModelType), full_output = True , disp=verbose, maxiter = maxiter, xtol=1e-6, ftol=1e-6)

		p[free] = popt
		if (warnflag < 1) & (p[1] > p[0]) & (p[5] < 2.*nyquist) & (p[5] > 0.1) & (p[4] > 0.05):
			p[4:6] = np.abs(p[4:6])
			numax = math.fabs(p[5])
			delta_nu = math.fabs(p[4])
			status = 3

			if(plot):
				plt.figure(2)
				plt.plot(f,BGModel(f,p,nyquist,norders,ModelType),'k', label=("step 4")) 
				plt.legend(loc=1)
				#ou plt.plot(f,BGModel(f,p,nyquist,norders,ModelType),'k', label=("Oscillation (Universal Pattern)"), linewidth=3)


			# tau_gran > tau_act : the granulation and activity components must be swapped 
			if (activity==True) & (p[2] > p[8]) :

				if(verbose): print "Permutation granulation-activite"

				tmp = p[1:3].copy()
				p[1:3] = p[7:9]
				p[7:9] = tmp
				tmp = perror[1:3].copy()
				perror[1:3] = perror[7:9]
				perror[7:9] = tmp

				# We need to recompute the fit in order to derive correctly the parameters error
				(popt,likelihood,direct,iter,funcalls,warnflag) = scipy.optimize.fmin_powell(BGModel_likelihood_powell, p0[free], args= (f[ind],PSD[ind],free,p0,nyquist,norders,ModelType), full_output = True , disp=verbose, maxiter = maxiter, xtol=1e6, ftol=1e-6)

				print "PERMUTATION GRAN-ACT"	# pour alerter

				p[free] = popt
				if (warnflag < 1) & (p[1] > p[0]) & (p[5] < 2.*nyquist) & (p[5] > 0.1) & (p[4] > 0.1):
					p[4:6] = np.abs(p[4:6])
					numax = math.fabs(p[5])
					delta_nu = math.fabs(p[4])
					status = 3

					if(plot):
						plt.figure(2)
						plt.plot(f,BGModel(f,p,nyquist,norders,ModelType),'k', label=("step 4-permut")) 
						plt.legend(loc=1)
						#ou plt.plot(f,BGModel(f,p,nyquist,norders,ModelType),'k', label=("Oscillation (Universal Pattern)"), linewidth=3)

				else:
					status = 2
					if(verbose): print 'convergence failed'


		else:
			if(verbose): print 'convergence failed'



	if (status == 3):

		ln_sigma2_gran = 0.
		if(ModelType == 1):
			if(p[3]>1.):
	                	ln_sigma2_gran = p[1] - p[2] + math.log (0.5) -math.log(p[3]*math.sin(math.pi/p[3]))
		elif(ModelType == 2):
			ln_sigma2_gran = p[1] - p[2] - math.log(math.pi)
		

		if(verbose):
			print
			print 'Final optimal parameters:', BGModel_log2lin(p,inv=False)	
			print "ln_sigma2_gran: %f" % ln_sigma2_gran	
			print "final likelihood: %f" % likelihood			

	
		if plot == True :		#fit computed

			p2 = p.copy()
			p2[6] = -10. 		# an envelop with a very small amplitude	
			background = BGModel(f,p2,nyquist,1,ModelType)

			env_comp = p.copy()
			env_comp[0]=-10.			#constante=0
			env_comp[1]=-10.			#Puissance de la granulation=0
			if (activity): env_comp[7]=-10.		#Puissance de l'activite=0
			env_profil = BGModel(f,env_comp,nyquist,norders,ModelType)

			gran_comp = p.copy()
			gran_comp[0]=-10
			gran_comp[6]=-10
			if (activity): gran_comp[7]=-10.
			gran_profil = BGModel(f,gran_comp,nyquist,norders,ModelType)

			if (activity): 
				act_comp = p.copy()
				act_comp[0]=-10
				act_comp[1]=-10
				act_comp[6]=-10
				act_profil = BGModel(f, act_comp,nyquist,norders,ModelType)

	
			plt.figure(3)
			plt.clf()
			plt.xlabel('Frequency [muHz]')
			plt.ylabel('Power [ppm2/muHz]')
			plt.title( ('Star ID: %i') % (StarID))

			window_len = int(math.ceil(delta_nu/f[0]/8.))		# Window_len should be an odd integer
			if window_len%2 == 0: window_len = window_len+1
			PSD_smoothed = smooth(PSD, window_len, window='hanning')
			fit_smoothed = smooth(BGModel(f,p,nyquist,norders,ModelType), window_len, window='hanning')

		        #plt.plot(f,PSD,'r', label=("Raw PSD"))
		        plt.plot(f,PSD,'grey', label=("raw PSD"))
		        plt.plot(f,PSD_smoothed,'k', label=("smoothed PSD"), linewidth=2)
		       	#plt.plot(f,fit_smoothed,'r', label=("Smoothed final fit"), linewidth=2)	#fit lisse
		       	#plt.plot(f,BGModel(f,p,nyquist,norders,ModelType),'k', label=("final fit"), linewidth=2)	#fit non lisse
			#plt.plot(f,np.zeros(f.size) + math.exp(p[0]),':g', label=("constant"), linewidth=2)			#, linewidth=3
			#plt.plot(f,env_profil,'-b', label=("universal pattern"), linewidth=1)	
			#plt.plot(f,gran_profil,'--g', label=("granulation"), linewidth=2)
			#if (activity): plt.plot(f,act_profil,'-.g', label=("activity"), linewidth=2)
			#plt.legend()
			plt.loglog()

	

			"""
			#-------------------------------- spectre theorique pour les simulations --------------------------
			#Parametres
			numax_th = 10.
			dt_th = 14.5*60	
			T_th = 1000.	
			mag_th = 12.0	
			seed_th = 1971	
			Dnu_th = 0.274*numax_th**0.757	
			DPI_th = 43.*Dnu_th**0.25	
			q_th = 0.17
			StarID_th = 0
			pn_ref_th = 34.	
			wn_ref_th = 0.
			mag_ref_th = 12.
			GST_th = 1
			navg_th = 2

			time_th,ts_th,f_th,ps_th,mpsd_th,par_th,seed_th = sls.gen_up(StarID, numax_th, dt_th, T_th, mag_th, DPI = DPI_th, q = q_th, seed = seed_th, pn_ref = pn_ref_th, wn_ref = wn_ref_th , mag_ref = mag_ref_th, verbose=False, GST = GST_th)


			a1k = 3382
			a1s = -0.609
			a1t = 0.

			a2k = 3710.
			a2s = -0.613
			a2t = -0.26

			b1k = 0.317
			b1s = 0.970

			b2k = 0.948
			b2s = 0.992

			c1 = 4.
			c2 = 4.
			numaxref= 3104.
			deltanusun = 135.
			teffsun = 5777.
			teff = 4800.
			mass = (numax/numaxref)**3. * (delta_nu/deltanusun)**(-4.) * (teff/teffsun)**(3./2.)

			a1 = a1k*numax_th**a1s*1**a1t
			a2 = a2k*numax_th**a2s*1**a2t

			b1 = b1k*numax_th**b1s
			b2 = b2k*numax_th**b2s


			xi1 = 2.*math.sqrt(2.)/math.pi
			tau1   = 1./(2.*math.pi)/b1

			xi2 = xi1
			tau2   = 1./(2.*math.pi)/b2

			h1 = a1**2/b1*xi1
			h2 = a2**2/b2*xi2


			#print par_th
			#par_th = (star_par,obs_par,osc_par,gran_par)
			#gran_par = {"h1": h1, "tau1": tau1, "alpha1": c1,"h2": h2, "tau2": tau2, "alpha2": c2,"taue": taue}
			g1_th = ssilib.plf(f_th,h1,tau1,c1)
			g2_th = ssilib.plf(f_th,h2,tau2,c2)

			plt.figure(30)
			plt.clf()
			plt.xlabel('Frequency [muHz]')
			plt.ylabel('Power [ppm2/muHz]')
			plt.title( ('Star ID: %i') % (StarID))

			plt.plot(f,PSD,'grey', label=("Raw PSD"))
		        plt.plot(f,PSD_smoothed,'k', label=("smoothed PSD"), linewidth=1)
			plt.plot(f_th,mpsd_th,'r', linewidth=2)
			plt.plot(f_th,g1_th,'--r', linewidth=2)
			plt.plot(f_th,g2_th,'--r', linewidth=2)
		       	plt.plot(f,BGModel(f,p,nyquist,norders,ModelType),'g', label=("final fit"), linewidth=1)	#fit non lisse
			plt.plot(f,np.zeros(f.size) + math.exp(p[0]),':g', label=("constant"), linewidth=2)			#, linewidth=3
			#plt.plot(f,env_profil,'-b', label=("universal pattern"), linewidth=2)	
			plt.plot(f,gran_profil,'--g', label=("granulation"), linewidth=2)
			plt.legend()
			plt.loglog()

			#--------------------------------------------------------------------------------------------------
			"""


			plt.figure(4)
			plt.clf()
			plt.xlabel('Frequency [muHz]')
			#plt.ylabel('Power [ppm2/muHz]')
			plt.title('PSD divided by the background model')

			PSDM = BGModel(f,p,nyquist,norders,ModelType)/background	# modeled spectrum
		        plt.plot(f,PSD/background,'k', label=("PSD"))
		       	#plt.plot(f,PSDM,'r', label=("Fitted model"))	#, linewidth=2

			"""
			nbin = max( (int(math.floor(p[4]/30./df)) , 1) )
			if ((nbin % 2 ==0) ):
				nbin = nbin +1 # to have an odd number (for the smooth function)
			PSDs = smooth(PSD/background,nbin) # smoothed spectrum

			PSDMs = smooth(PSDM,nbin) # modeled spectrum, smoothed
			(nu,ell,order,height,gamma) = UP.universal_pattern_modes(df, p[4], p[5] , math.exp(p[6]) ,  verbose = False , teff=4800. , DPI = -1. , q = 0 )


			plt.figure(14)
			plt.clf()
			plt.plot(f,PSDs,'k')
			plt.plot(f,PSDMs,'b')
			i = 0
			for l in ell:
				if (l==0):
					plt.plot( [nu[i],nu[i]],[0,np.max(PSDs)*1.2],'r--' )
				elif (l==1):
					plt.plot( [nu[i],nu[i]],[0,np.max(PSDs)*1.2],'g--' )
				elif (l==2):
					plt.plot( [nu[i],nu[i]],[0,np.max(PSDs)*1.2],'m--' )
				elif (l==3):
					plt.plot( [nu[i],nu[i]],[0,np.max(PSDs)*1.2],'c--' )


				i = i +1
			"""	


	""""""""""""""""""""""""""""""""""""" 5th step - error calculation """""""""""""""""""""""""""""""""""""

	if(status == 3):

		pb_gran = False
		pb_osc = False
		ln_sigma2_gran_err = 0.

		if(verbose):
			print 
			print '--- last step: Computing the errors'

		# Error calculation
		Hessian,covar,cdtn,ok = hessian(BGModel_likelihood, p , (f[ind],PSD[ind],nyquist,norders,ModelType) , free = free, verbose = verbose )

		if(ok):
			status = 4
			for i in range(p.size):
				if( (covar[i,i] > 0.) & (free[i]) ):
					perror[i] = math.sqrt(covar[i,i]) * ( free[i] == True )
				else:
					if ( free[i]):
					        if(verbose): print ("WARNING ! free parameter # %d with a negative or zero variance !") % (i)
						perror[i] = -math.sqrt(-covar[i,i])
						ok = False
						if (i<7) : status = 3
						if (i>=7): status = 5
						if (i==1) | (i==2) | (i==3) : pb_gran=True	# some probleme with the granulation
						if (i==4) | (i==5) | (i==6) : pb_osc=True	# some probleme with the oscillations

			if(verbose & (ok==False)):
				print 
			        print 'Hessian matrix:'
			        for i in range(len(p)):
			                if(free[i]):
			                        print i,":", Hessian[i,free]
		


		if ( ( (pb_gran==True) | (pb_osc==True) | (p[3] > 5.0) ) & (refit==False) & (activity==True) ): # We have to recompute the BG
			#le test sur refit evite de refaire plus d'une fois le fit, ce qui evite une boucle infinie
			recompute = True
			compt_refit = 1

	 	# calculation of ln_sigma2_gran_err
		if ( (free[1]) & ( (p[3]>1.) | (ModelType==2) ) ):
			p2 = p.copy()
			p2[1] = ln_sigma2_gran
			Hessian,covar2,cdtn2,ok = hessian(BGModel_likelihood, p2 , (f[ind],PSD[ind],nyquist,norders,ModelType,True) , free = free, verbose = verbose )
			if( covar2[1,1] >0.):
				ln_sigma2_gran_err = math.sqrt(covar2[1,1])
			else:
				ln_sigma2_gran_err = -math.sqrt(-covar2[1,1])
				if(verbose):
					print ("WARNING ! parameter sigma2_gran presents a negative or zero variance ")

		p2 = p.copy()
		p2[6] = -10. # an envelop with a very small amplitude	
		background = BGModel(fprime,p2,nyquist,1,ModelType)
		Bmax = BGModel(np.array([numax]),p2,nyquist,1,ModelType) [0]


		if(verbose):
	 		print 'perror:',perror
			print ("ln_sigma2_gran_err = %f") % (ln_sigma2_gran_err)
			print
			print ('final optimal numax = %f +/- %f') % (numax, perror[5])
			print ('final optimal delta_nu = %f +/- %f') % (delta_nu, perror[4])
			print "respect de la loi d'echelle", (delta_nu >= (0.280*numax**0.747)/1.3) & (delta_nu <= (0.280*numax**0.747)*1.3)
			print "numax loi echelle", (math.fabs(delta_nu)/0.274)**(1/0.757)
			print "Ratio S/B =", math.exp(p[6])/Bmax


	# If one fit did not succeed
	if (status < 3):
		if(verbose): print "background fit did not succeed"
		likelihood = -1
		Bmax = -1	
		background = -1	
		ln_sigma2_gran = 0.
		ln_sigma2_gran_err = -1	
		if (refit==False) & (activity==True) : 
			recompute = True
			compt_refit = 1
		else : recompute = False


        return (likelihood,p,perror,free,status,cdtn,Bmax,background,numax_gran,ln_sigma2_gran,ln_sigma2_gran_err,compt_avAct,compt_ssAct,compt_uncer,compt_refit,recompute,step2_numax_Dnu,step2_Dnu_numax, step2_step1) 


def usage():
    print "usage : mle_up name.xml"
    print "      : mle_up -v"
    print "Options:"
    print "-p : prompt the user (not compatible with multiprocessing)"
    print "-M <n> : do multiprocessing with <n> processors"
    print "-l <file> : process the stars corresponding to the IDs listed in <file>"
    print "-r <file> : resume a previous calculation stored in the binary named <file>"

if(len(sys.argv)<2):
    usage()
    sys.exit(2)
try:
    opts,args = getopt.getopt(sys.argv[1:],"hpM:l:r:v")

except getopt.GetoptError, err:
    print str(err)
    usage()
    sys.exit(2)

nproc = 1
prompt = False
flist = ''
frestore = None
for o, a in opts:
    if o == "-h" :
        usage()
        sys.exit(1)
    elif o == "-p":
        prompt = True
    elif o == "-M":
        nproc = int(a)
    elif o == "-l":
        flist = a
    elif o == "-r":
        frestore = a
    elif o == "-v":
        print __version__
        sys.exit(1)
    else:
        print "unhandled option %s" % (o)
        sys.exit(1)


nargs = len(args)
if nargs > 1 :
    print "too many arguments"
    usage()
    sys.exit()

if nargs < 1 :
    print "missing arguments"
    usage()
    sys.exit()

config=args[0]

# lecture des fichiers binaires des etoiles a etudier
doc = minidom.parse(config)
elem= doc.getElementsByTagName('pp_output')
pp_output= elem.item(0).firstChild.nodeValue
StarIDList=GF.StarID_List(pp_output)

NScan = 2 	# Number of autocorrelation scans
c = 2.**0.1	# sqrt(sqrt(2))  # ratio for geometric progression
ACFRes 	 = str(GF.getParm(config,"ACFRes"))
ACFThreshold   = float(GF.getParm(config,"ACFThreshold"))
NDnu     = float(GF.getParm(config,"NDnu"))		
Nu0      = float(GF.getParm(config,"Nu0"))
Nu1      = str(GF.getParm(config,"Nu1"))
Activity = bool(int(GF.getParm(config,"Activity")))              	

if Nu1=='nyquist': 
   NFilter=0
else:
   Nu1 = float(Nu1)
   NFilter = int (round( (math.log(Nu1)-math.log(Nu0))/math.log(c) ) )  + 1
   print('Number of filters used for the autocorrelation: {}'.format(NFilter))
print('Resolution used for the ACF method during the 2nd iteration: {}'.format(ACFRes))
#print('Number of filter during the 2nd iteration: {}'.format(NFilter2))
print('Number of autocorrelation scans: {}'.format(NScan))
print('Delta nu multiplication factor: {}'.format(NDnu))
print('First frequency taken in [muHz]: {}'.format(Nu0))
print('Limit frequency of numax/deltanu research [muHz]: {}'.format(Nu1))
print('Activity component for the background fit: {}'.format(Activity))
print('Common ratio for geometric progression: {}'.format(c))


N_object = len(StarIDList)	#Number of object that we want to analyse
if (N_object == 0):
	print 'No stars with pre-processed light-curves ! nothing to do !'
	sys.exit(1)

if (flist != ''):
# The ID of the stars we want to process are taken from the file <flist> 
    ID = np.loadtxt(flist)
    NewStarIDList = []
    for i in range(N_object):
        k = np.where( ID == StarIDList[i][0])[0]
        if ( len(k) > 0 ):
            NewStarIDList.append(StarIDList[i])
    n = len(NewStarIDList)
    if ( n > 0):
        print ('Of the %d stars listed in %s, %d are available') % (ID.size,flist,n)
        StarIDList = NewStarIDList
        #if (prompt and (nproc < 2) ):
        #    s = raw_input("Must we process these stars,  yes (y) or not (n) ?\n")
        #    if (s.lower() == 'n'  ):
        #        sys.exit( 0)
    else:
        print ('None of the stars listed in  %s are available') % (flist)
        sys.exit(1)

 
N_object = len(StarIDList)	#Number of object that we want to analyse

graph = (0,0,0,0,0,0)
#plt.ioff()
plot = False
interactive = False
if ( nproc > 1):
    verbose = False
    a=1
else:
    verbose = True
    

if (prompt and (nproc < 2) ):
    s = raw_input("Do the plots,  yes (y) or not (n) ?\n")
    plot = (s.lower() == 'y')

    if (plot):
        #[0]:light curve; [1]:PSD ; [2]:Autocorrelations curves ; [3]:nu vs Maxima ; [4]:Tau vs Maxima ; [5]: PSD divided by filters
        graph = (0,0,0,0,0,0)
	import matplotlib.pyplot as plt
        plt.ioff()
    s = raw_input("Interactive mode, yes (y) or not (n) ?\n")
    interactive = s.lower() == 'y'
elif ( prompt and (nproc >= 2) ):
    print 'Interactive mode and plot are not compatible with multiprocessing' 

if (not plot):
	import matplotlib
	matplotlib.use('Agg')

nvalues = 49
results = np.zeros( (N_object,nvalues) )
results[:,1] = -100 # by default status=-100 meaning that the associated star has not be processed

if (frestore != None):
        print 'restoring data file  ',frestore
        datatmp = np.load(frestore)
        print ("Among %i stars, %i have already been processed") % (datatmp.shape[0], (np.abs(datatmp[:,1]+ 100.) >0.).sum())

def process_one_star(Star):

    StarID = Star[0] # Star ID
    Origin = Star[1] # Star Origin
    if(verbose):
        print '##########################################################################'
        print 'StarID: ', StarID
        
    if(frestore != None):
        match =  (np.abs(StarID - datatmp[:,0]) < 1e-5) & ( np.abs(datatmp[:,1]+ 100.) > 1e-5) # we search this star among the stars already processed (status !=-100)
	nmatch = match.sum()
	if(nmatch > 1):
	        print ('Warning ! more than one occurence of the star ID %9.9i in the restored file %s') % (StarID,frestore)
	if ((nmatch > 0) & verbose):
	        print ('star ID %9.9i already processed, restoring the data from %s') % (StarID,frestore)
	if (nmatch > 0):
	        return datatmp[match,:][0,:]

    record=Record(config)
    record.read_app(StarID, Origin)		#lire les donnees generees par le programme de pre-processing
    passage = 1

    nyquist = np.max(record.Freq)*1e6		# in muHz

    y = np.roll(record.Freq,1)
    dify = record.Freq-y   
    freq_res = np.median(dify)*1e6		# Frequency resolution

    z = np.roll(record.Time,1)
    difz = record.Time-z   
    time_res = np.median(difz)			# Frequency resolution

    if (verbose):
	print '**************************************************************************'
        print 'Computing the ACF'
    Amax, numax_Auto, err_numaxA, deltanu, err_deltanu, flag = auto.autocorrelation_list(record.Time, record.Flux, record.Freq, record.PSD, NFilter, Nu0, c, NDnu, Nu1, NScan,graph,passage,Origin,StarID)

    if(verbose):
        print ('Results of the ACF: numax=%f, deltanu=%f, Amax=%f') % (numax_Auto, deltanu, Amax)
    if(Amax < ACFThreshold):
        if(verbose):
            print ('ACF peak below the threshold = %f, we skip this star') % (ACFThreshold)
        data = np.zeros(nvalues)
        data[0]  = StarID
        data[1]  = -102.
        data[27] = Amax
        data[28] = numax_Auto
        data[30] = deltanu
        data[31] = err_deltanu
        data[47] = freq_res
        data[48] = nyquist
        return data

    if (verbose):
	print 
	print '**************************************************************************'
        print 'Computing the Background and oscillations'
    (likelihood,popt,perror,free,status,cdtn,Bmax,background,numax_gran,ln_sigma2_gran,ln_sigma2_gran_err, avAct, ssAct, uncer, refit, recompute, step2_numax_Dnu, step2_Dnu_numax, step2_step1)  = background_mle_fit (record.Freq , record.PSD , numax_Auto, deltanu, Nu0 , Nu1, Origin, StarID, nyquist, activity=Activity, plot=plot, verbose=verbose, refit=False)

    if (recompute==True):
	    if (verbose):
		print 
		print " => Probleme with the granulation, recomputing the background and oscillations without activity component"
	    (likelihood,popt,perror,free,status,cdtn,Bmax,background,numax_gran,ln_sigma2_gran,ln_sigma2_gran_err, avAct, ssAct, uncer, refit, recompute, step2_numax_Dnu, step2_Dnu_numax, step2_step1)  = background_mle_fit (record.Freq , record.PSD , numax_Auto, deltanu, Nu0 , Nu1, Origin, StarID, nyquist, activity=False, plot=plot, verbose=verbose, refit=True)
	    refit = 1


    freeflag = 0
    for i in range(len(free)):
	    if(free[i]):
	        freeflag += (2**i)
    numax = popt[5]
    
    ltaue = -100.
    ltaue_err = -100.
    if( (popt[3] > 0.) & (popt[3] < 100.)  ):
	if(ModelType == 1):
		ltaue = math.log( ssilib.eft(math.exp(popt[2]),popt[3],ssilib.plf) )  
	elif(ModelType == 2):
		ltaue = math.log( ssilib.eft(math.exp(popt[2]),popt[3],ssilib.pef) )  
	ltaue_err = perror[2]
    if(verbose):
        print ("taue, tau [s]: %e , %e ") % (math.exp(ltaue)*1e6  , math.exp(popt[2])*1e6)
	print 
    
    passage = 2
    Amax2=0.
    numax_Auto2=0
    err_numaxA2=0
    deltanu2=0
    err_deltanu2=0
    flag2=0

    # If background_mle_fit succeed we compute the second ACF
    if (status >= 3) & (numax < 2.*nyquist) & (numax > 0.1) & (popt[4] > 0.1) & (perror[4]!=0) & (perror[5]!=0):		
	    if(verbose):
		print '**************************************************************************'
		print 'Computing the 2sd ACF'

	    PSD = record.PSD		# PSD = record.PSD/background	# pour ne pas prendre en compte le BG
	    Nu02 =  numax*2/3.
	    Nu12 = numax*4/3.
	    if ACFRes == 'resolution': ACFres = freq_res
	    else : ACFres = float(ACFRes)
	    NFilter2 = int(round((Nu12-Nu02)/ACFres)) + 1
	    c2 = ACFres			#New arithmetic progression : two times smaller than the resolution of the spectrum
	    NScan2 = 1

	    try:
		Amax2, numax_Auto2, err_numaxA2, deltanu2, err_deltanu2, flag2 = auto.autocorrelation_list(record.Time, record.Flux, record.Freq, PSD, NFilter2, Nu02, c2, NDnu, Nu12, NScan2,graph,passage,Origin,StarID)
	    except:
		if (verbose):
		    print '... did not succeed'
	    if(verbose):
		print ('Results of the 2sd ACF: numax = %f, deltanu = %f, Amax = %f') % (numax_Auto2, deltanu2, Amax2)
		print "numax loi echelle", (math.fabs(deltanu2)/0.274)**(1/0.757)
		print "respect de la loi d'echelle", (deltanu2 >= (0.280*numax_Auto2**0.747)/1.3) & (deltanu2 <= (0.280*numax_Auto2**0.747)*1.3)
		print 
	        print 'StarID: ', StarID
	
    else: 
	    if (verbose):  print 'Error in the background and oscillations fit. The 2sd ACF is not computed'


    if (plot):
	    plt.show(block=False)
	    plt.draw()


    return (StarID,status,likelihood,freeflag,popt[0],perror[0],popt[1],perror[1],popt[2],perror[2],popt[3],perror[3],popt[6],perror[6],popt[5],perror[5],\
	    popt[4],perror[4],popt[7],perror[7],popt[8],perror[8],popt[9],perror[9],cdtn,ltaue,ltaue_err,Amax, \
	    numax_Auto, numax_gran, deltanu, err_deltanu, Amax2, numax_Auto2, err_numaxA2,deltanu2, err_deltanu2,Bmax, \
	    ln_sigma2_gran,ln_sigma2_gran_err, avAct, ssAct, uncer, refit, step2_numax_Dnu, step2_Dnu_numax, step2_step1, freq_res, nyquist)



def process_shell(Star):
    # coquille utilise pour le multiprocessing
    results = np.zeros( (nvalues) )
    results[0] = Star[0]
    results[1] = -101.
    try:
        print ("processing star ID=%d") % (Star[0])
        sys.stdout.flush()
        results[:] = process_one_star(Star)
        print ("Star ID=%d has been processed") % (Star[0])
    except KeyboardInterrupt:
        return results
    except SystemExit:
	sys.exit(-1)
    except:
        traceback.print_exc()
        print ('some errors occur while processing star ID=%d') % (Star[0])
    finally:
        sys.stdout.flush()
    return results

    
# multi-processing
if (nproc>1):
    print('start :{}'.format(datetime.datetime.now()))
    nproc = min(nproc,N_object)

    # define the handler function
    def term_handler(signum, frame):
    	print "TERM/INT signal received. Exiting this job."
   	sys.exit(-1)
    signal.signal(signal.SIGTERM, lambda signum, stack_frame: sys.exit(1))

    def init_worker():
	signal.signal(signal.SIGINT, term_handler)
	signal.signal(signal.SIGTERM, term_handler)

    pool = multiprocessing.Pool(nproc,init_worker)

    outputs =  [pool.apply_async(process_shell,(StarIDList[i],)) for i in range(N_object)]

    pool.close()

    def finish(signum, frame):
        print "TERM/INT signal received. Exiting. The results are backup"
        pool.terminate()
        pool.join()
	# collecting the results
    	i = 0
    	for output in outputs:
		if (output.ready()):
        		results[i,:] = output.get()
        	i = i + 1
        np.save(re.sub('\.xml','',config)+'_mle_UP_BACKUP',results)
        sys.exit(-1)

    signal.signal(signal.SIGTERM,finish)
    signal.signal(signal.SIGINT,finish)

    # waiting for the completion
    count = 0
    countp = 0
    while ( count < N_object):
    	count = 0 
	for output in outputs:
		if (output.ready()):
			results[count,:] = output.get()
			count = count + 1
	time.sleep(1)
	if( count > countp +  float(N_object)*1e-2 ):
		print ("%5.0f %% completed") % (float(count)/float(N_object)*100.)
		sys.stdout.flush()
		countp = count
		np.save(re.sub('\.xml','',config)+'_mle_UP_BACKUP',results)
    pool.join()
    # collecting the results
    i = 0 
    for output in outputs:
    	results[i,:] = output.get()
	i = i + 1

    print('end :{}'.format(datetime.datetime.now()))
            
# mono-processing
else: 

    stop = False

    for i in range(0,N_object):
	results[i,0] = StarIDList[i][0] # Star ID
        print i+1,'/',N_object
    	try:
        	results[i,:nvalues] = process_one_star(StarIDList[i])
        except KeyboardInterrupt:
            print 'Keyboard interrupt'
	    np.save(re.sub('\.xml','',config)+'_mle_UP_BACKUP',results)
            sys.exit(1)
        except:
            traceback.print_exc()
            results[i,1] = -101.
            print ('some errors occur while processing star ID=%d') % (StarIDList[i][0])
        
        if (plot):
            plt.show(block=False)
            plt.draw()  
        if (interactive):
                s = raw_input("To continue type ENTER\nTo stop type 'q' or 's'\n")
                if( s.lower() == 'q' or s.lower() == 's'):
                        stop = True
        print ""
        print ""
        if(stop):
                break


print 'finished !'
k = np.where( results[:,1] >= 0.  )[0]
print ("%d stars among %d have been sucessfully processed") % (len(k),N_object)

if (interactive):
        s = ''
        while ( s !='y' ):
                s = raw_input("Write the results [y/n] ? ")
                s = s.lower()
                if (s == 'n'):
                        sys.exit(0)

np.save(re.sub('\.xml','',config)+'_mle_UP',results)

# data[:,i]:
#  0 : ID
#  1 : status
#  2 : l = - log (likelihood)
#  3 : flag of the free parameters (the ith parameter is free if the ith bit is 1)
#  4 : log white noise [ppm^2/muHz]
#  5 : log white noise err
#  6 : log height gran [ppm^2/muHz]
#  7 : log height gran err
#  8 : log tau gran [10^6 s]
#  9 : log tau gran err
# 10 : slope gran
# 11 : slope gran err
# 12 : log height up [ppm^2/muHz]
# 13 : log height up err
# 14 : numax up [muHz]
# 15 : numax up err
# 16 : deltanu up [muHz]
# 17 : deltanu up err
# 18 : log height act [ppm^2/muHz]
# 19 : log height act err
# 20 : log tau act [10^6 s]
# 21 : log tau act err
# 22 : slope act
# 23 : slope act err
# 24 : Hessian matrix condition number
# 25 : log taue gran [10^6 s]
# 26 : log taue gran err
# 27 : Amax
# 28 : numax guess from ACF [muHz]
# 29 : numax guess from tau_gran [muHz]
# 30 : delta nu ACF  [muHz]
# 31 : delta ACF  err 
# 32 : Amax 2
# 33 : numax ACF 2 [muHz]
# 34 : numax ACF 2 err 
# 35 : delta nu ACF 2 [muHz]
# 36 : delta nu ACF 2 err 
# 37 : Bmax [ppm^2/muHz]
# 38 : log sigma2_gran [ppm^2]
# 39 : log sigma2_gran err
# 40 : number of objects with Activity 
# 41 : number of objects without Activity 
# 42 : number of objects whose the Activity is not sure
# 43 : number of objects refit without Activity
# 44 : step2_numax_Dnu
# 45 : step2_Dnu_numax
# 46 : step2_step1
# 47 : resolution frequentiel [muHz]
# 48 : Nyquist frequency  [muHz]
