'''

   	This file is part of the Stellar Seimic Indices pipeline

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

	Copyright (C) 2016 by R. Samadi 
'''
import numpy as np
from derivee2xy import *
import matplotlib.pyplot as plt



"""
(hessian,covar,cdt,ok) = bao_i_hessian(name,param,free=None)

compute the hessian for a given Maximum Likelihood function

INPUTS:
name 		the Maximum Likelihood function 
param 		optimal values of the parameters
args            arguments transmitted to the function (tuple)
free            free[i]= 1 for the free parameters
	
OUTPUTS:        
covar		covariance matrix
hessian         Hessian matrix
cdt             condition number
ok              flag,  ok=False in case of failure

Adapted by R.S. from the IDL routine developped by Thierry Appourchaux
"""


def hessian(name,param,args,free=None,verbose=False):
        
        # compute the matrix elements
	Ndim = len(param)
	hessian = np.zeros( (Ndim,Ndim) )
	if (free == None):
	      free  = np.ones(Ndim,dtype=np.bool)
	      nfree = ndim
        else:
                nfree = np.sum(free)
        free2d = np.zeros((Ndim,Ndim),dtype=np.int)
        covar = np.zeros( (Ndim,Ndim) )

	for i in range(0,Ndim):
	        for j in range(i,Ndim):
			if(free[i] & free[j]):
                           hessian[i,j] = derivee2xy(name,param,i,j,args)
                           hessian[j,i] = hessian[i,j]
        for i in range(0,Ndim):
                for j in range(0,Ndim):
                        free2d[i,j] =  free[i] & free[j]
        j = np.where(free2d)
        temp = ((hessian[j]).copy()).reshape( (nfree,nfree) )
        cdtn = 0.
        try:
                # singular value decomposition (SVD): A = U S V
                (U,s,V) = np.linalg.svd(temp)
                #  diagonal matrix that will contain the singular values (eigenvalues)
                Si = np.zeros( (nfree,nfree) )
                # we truncate the matrix 
                limit = max(s)*1e-10
                for i in range(nfree):
                        if(s[i] > limit):
                                Si[i,i] =  (1./s[i])  # we retain only eigenvalue s > limit
                        else:
                                Si[i,i] = 0.
                if( min(np.abs(s)) != 0. ):
                        cdtn = max(np.abs(s))/min(np.abs(s)) # condition number
                        if(verbose):
                                        print ('Hession matrix condition number: %g' ) % (cdtn  )    
                else:
                        if(verbose):
                                print 'there is a null eigenvalue'
                # matrice inverse: A^(-1) = V^t S^(-1) U^t 
                temp = np.dot(np.transpose(V),np.dot(Si,np.transpose(U)))
#                plt.figure(0)
#                plt.clf()
#                plt.plot(s)
#                plt.plot([0,nfree-1],[limit,limit])
#                plt.semilogy()
        #       temp = np.linalg.inv(temp)
	except:
	        traceback.print_exc()
		print 'the Hessian matrix can not be inverted'
		ok = False
		pass
	else:
                covar[j] =  temp.reshape( (nfree*nfree) )
                ok = True
        return (hessian,covar,cdtn,ok)
