'''
    Auto-correlation function 

	This file is part of the Stellar Seimic Indices pipeline

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

	Copyright (C) 2016 by R. Peralta 
'''


#INPUTS : 
#   - N : delta nu multiplication factor
#   - nu : scalar [microHz] frequency of the center of the cosine filter
#   - delta_nu : scalar [microHz] width in freqeuncy of the cosine filter
#   - f : PSD frequency vector
#   - p : Power Spectrum Density (PSD) vector

#OUTPUTS :
#   - time : autocorrelation time vector
#   - auto_norm : autocorrelation normalized vector
#   - p_filtered : PSD vector filtered by the cosine filter

import fftw3
import numpy as np
import math
        
def autocorrel_func (N, nu, delta_nu, delta_nu_folded, f, p, fftplan, p_filtered , p_ft):
    
    nu=nu*1e-6                   #[Hz]
    delta_nu=delta_nu*1e-6       #[Hz]
    delta_nu_folded = delta_nu_folded*1e-6
    
    #Cosine filter. Area filter selection : [nu[i]-(N*delta_nu)/2 : nu[i]+(N*delta_nu)/2]
    nf = len(f)   
    index =  (np.where( (f >= (nu - (N*delta_nu)/2.)) & (f <= (nu +(N*delta_nu)/2.)) )) [0]
    cos_filter = np.zeros(nf)
    cos_filter [index] = (np.cos( ((2.*math.pi)/(N*delta_nu))*(f[index]-nu) )+1.)/2.	#period = N*delta_nu; shift = nu
    

    #PSD*cos
    p_filtered[1:nf+1] = p*cos_filter
    #p_filtered[nf+1:2*nf+1] = p*cos_filter


    #FFT-1(PSD*cos)
    fftw3.execute(fftplan)
	#p_filtered is the data array input ; p_ft, the output of fftw3
	#http://www.fftw.org/doc/Complex-One_002dDimensional-DFTs.html

    autocorrel = abs((p_ft[0:nf+1]**2))
	#For DFT performance : If A = fft(a, n), then A[0] contains the zero-frequency term (the mean of the signal), 
	#which is always purely real for real inputs. Then A[1:n/2] contains the positive-frequency terms, 
	#and A[n/2+1:] contains the negative-frequency terms, in order of decreasingly negative frequency.
	#(http://docs.scipy.org/doc/numpy/reference/routines.fft.html)


    #Normalisation of the peaks autocorrelation of : (for more details see Mosser & Appourchaux 2009 - (2))
    #Amplitude
    A_star = autocorrel[1:nf+1]/autocorrel[0]

    #Sigma 
#    dnu = f[0]              #frequency resolution = 1/T  => f[1]-f[0] or 1/time[len(time)-1] gives the same result
    Nh = max(1 , len(index)  ) #Number of frequency bins selected in the filter
#    Nh = N * delta_nu / dnu      #Number of frequency bins selected in the filter
    sigma = 3./(2*Nh)            #Noise unit. Cf. equation (5) Mosser and Apourchaux (2009)
    A = A_star/sigma             #Sigma normalisation
 

    #Autocorrelation time vector
    df = f[1]-f[0]        
    dt =  1./(2.*nf*df)
    time = np.arange(1,nf+1) * dt


    #Thresholds : limitations of autocorrelation betwenn : [2/deltanu +- (N/delta_nu)/2]
#    gamme = np.where ( ((time >= (2/delta_nu)/1.3) & (time <= (2/delta_nu)*1.3)) | ((time >= (2/delta_nu_folded)/1.3) & (time <= (2/delta_nu_folded)*1.3))  )[0]
    gamme = np.where ( ((time >= (2/delta_nu)/1.3) & (time <= (2/delta_nu)*1.3))   )[0]

    return (time, A, p_filtered[1:nf+1], gamme, cos_filter) 
    #return (time, A, p_filtered[nf+1:2*nf+1], gamme, cos_filter) 






