#!/usr/bin/env python
# -*-coding:Utf-8 -*
'''

   	This file is part of the Stellar Seimic Indices pipeline

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

	Copyright (C) 2016 by M. Abed and R. Samadi
'''

import pickle
import pyfits
import numpy as np
import math
#from numpy import *
import pynfftls
from xml.dom import minidom
from  scipy.optimize import curve_fit
import datetime
import os,sys, traceback
import GestionFiles.GF as GF
import JumpFilter
from smooth_func import *
from matplotlib.pyplot import *

try:
 from Files import Files
except:
 from Struc.Files import Files
try:
 from Indice import Indice
except:
 from Struc.Indice import Indice

def checkbit(value,bitorder):
    mask = 1 << bitorder
    return (np.int32(value) & mask ) == mask
 
def running_median(signal,win,frac, std=False):
    step = max(int(win*frac),1)
    n = len(signal)
    win = min(n,win)
    rmed = np.zeros(n)
    if (std):
     rstd = np.zeros(n)
    i0 = win/2
    for i in range(i0,n-i0,step):
     j = i - i0
     rmed[i] = np.median(signal[j:j+win])
     rmed[i+1:i+step] =  rmed[i]
     if (std):
         mean = np.sum(signal[j:j+win])/win
         if (win >1):
             rstd[i] = np.sqrt(np.sum( (signal[i-i0:i+win]-mean)**2)/(win-1))
             rstd[i+1:i+step] =  rstd[i]
    for i in range(0,i0):
     rmed[i] = rmed[i0]
     if (std):
         rstd[i] = rstd[i0]  
    for i in range(n-i0,n):
     rmed[i] = rmed[n-i0-1]
     if (std):
         rstd[i] = rstd[n-i0-1]  
    if (std):
     return rmed , rstd
    return rmed

def smooth_corot_exo_lc(Time,LC,Flag,Size,Sampling,verbose=False,type=1):
    # Size : window size : in seconds
    # Type : type of smooth, type=1 for a running median and type = 2 for running mean 
    n = Time.size
    Smoothed = np.zeros(n)
    if (type==1):
	StdDev = np.zeros(n)	

    dt = Time[1:]-Time[0:-1]
    eps = 1e-3
    F32 = np.abs(Sampling -32.) < 1e-10

    F512 = np.abs(Sampling -512.) < 1e-10

    wl = int( np.round(Size/32.) )# window_len
    wl = wl + (wl %2 ==0)*1 # window_len should must  an odd integer
    j = np.where( (Flag == True ) & (F32) ) [0]
    nt = np.sum( (Flag == True) )
    n32 = len(j)
    wl = min([n32 - (n32  % 2 ==0)*1 ,wl])
    if ( n32 > 0):
     if (verbose):
         print ('n 32s= %d, frac= %f, window size= %d') % (n32, float(n32)/float(nt) ,wl)
     if (type==1):
         Smoothed[j],StdDev[j]  =  running_median ( LC[j] , wl , 0.2 , std=True)
     else:
         Smoothed[j]  =  smooth( LC[j] , window_len=  wl ,window='hanning')
   

    wl = int( np.round(Size/512.) ) # window_len
    wl = wl + (wl %2 ==0)*1 # window_len should must  an odd integer
    j = np.where( (Flag==True) & (F512) ) [0]
    n512 = len(j)
    wl = min([n512 - (n512  % 2 ==0) *1 ,wl])
    if ( n512 > 0):
     if (verbose):
        print ('n 512s= %d, frac= %f, window size= %d') % (n512, float(n512)/float(nt) ,wl)
     if (type==1):
         Smoothed[j],StdDev[j] =  running_median ( LC[j] , wl , 0.2, std=True )
     else:
         Smoothed[j]  =  smooth( LC[j] , window_len= wl  ,window='hanning')

    if (verbose):
     if ( (n512>0) | (n32>0) ):
         nk = np.sum ((Flag==True) &  ( (F512) | (F32) ) )
         print ('discarded measurements= %d , fraction= %f') % (nt-nk,1.-float(nk)/float(nt) )

    if (type==1):
    	return (Smoothed, (Flag==True) & ( F32 | F512 ) , StdDev )
    else:
	return (Smoothed, (Flag==True) & ( F32 | F512 ) )


def smooth_lc(Time,LC,Flag,Size,typeFile,Sampling,NSampling,verbose=False,type=1):

    # Size : window size : in seconds
    # Type : type of smooth, type=1 for a running median and type = 2 for running mean 
    if ( (typeFile >=2) & (typeFile<=3) ): # exo channel
    	return smooth_corot_exo_lc(Time,LC,Flag,Size,Sampling,verbose=verbose,type=type)

    n = Time.size
    Smoothed = np.zeros(n)
    if (type==1):
        StdDev = np.zeros(n)	

    dt = Time[1:]-Time[0:-1]
    retain = np.zeros(n,dtype=np.bool)
    T = (np.abs(dt) < Size/2.) 
    retain[0:-1] = T
    retain[1:] = T
    nt = np.sum( (Flag == True) )
    wl = int( np.round(Size/NSampling) ) # window_len
    wl = wl + (wl %2 ==0)*1 # window_len should must  an odd integer
    j = np.where( (Flag==True) & (retain) ) [0]
    nr = len(j)
    wl = min([nr - (nr  % 2 ==0)*1 ,wl])
    if (type==1):
        Smoothed[j],StdDev[j] =  running_median ( LC[j] , wl , 0.2, std=True )
    else:
        Smoothed[j]  =  smooth( LC[j] , window_len= wl  ,window='hanning')

    if (verbose):
        if ( nr>0 ):
            nk = np.sum ((Flag==True) &  ( (retain)  ))
            print ('discarded measurements= %d , fraction= %f') % (nt-nk,1.-float(nk)/float(nt) )

    if (type==1):
        return (Smoothed, (Flag==True) & ( retain ) , StdDev )
    else:
        return (Smoothed, (Flag==True) & ( retain ) )

def spread_flag(Flag,win):
	n = len(Flag)
	Flag_out = np.zeros(n,dtype=np.bool)
	win = min(n,win)
    	i0 = win/2
    	for i in range(i0,n-i0):
     		j = i - i0
     		Flag_out[i] = np.sum(Flag[j:j+win])
	for i in range(0,i0):
     		Flag_out[i] = Flag_out[i0]
 	for i in range(n-i0,n):
     		Flag_out[i] = Flag_out[n-i0-1]

	return Flag_out


def detect_jumps(Time,SLC,STDEV,WinSizeSmooth,Sampling,threshold,verbose=False,plot=False):
    n = len(Time)
    diff = np.zeros(n)
    wl = int( np.round(float(WinSizeSmooth)/float(Sampling)) )# window_len
    wl = wl + (wl %2 ==0)*1 # window_len must be an odd integer
    diff[1:] =  (SLC[1:] - SLC[0:-1])
    sig = np.std(diff)   # *float(WinSizeSmooth)/float(Sampling)) 
    if (verbose):
	print ('sig= %f') %(sig)
    diff = np.abs(diff)
    threff =  threshold # *sig
    Flag = np.zeros(n,dtype=np.bool)
    Flag[1:] = spread_flag(diff[1:] > threff , wl )
    idx = np.where( Flag  ) [0]

    jumpin = np.zeros(n,dtype=np.bool)
    jumpout = np.zeros(n,dtype=np.bool)
    T = np.array(Flag,dtype=np.float)
    T = T[1:] - T[0:-1]
    jumpin[1:] = T > 0.
    jumpout[1:] = T < 0.
    locin = np.where( jumpin) [0]
    locout = np.where( jumpout) [0]
    if (verbose):
        print ("number of jumps = %d") % ( len(locin) )

    if (plot):
        import matplotlib.pyplot as plt
        plt.figure(100)
        plt.clf()
        plt.plot(Time,SLC)
 
        plt.figure(101)
        plt.clf()
        plt.plot(Time,diff)
        plt.plot([Time[0],Time[-1]],[threff,threff])
#	plt.plot(Time, threshold * stdev )
    njumps = len(locin)
    if (len(locout) < njumps):
	locout = np.append(locout,n-1)
    if (njumps   > 0):
        Amp = np.zeros(njumps)
        for i in range(njumps):
            Amp[i] = np.max( diff[locin[i]:locout[i]+1]  )
        # the jump are sorted by decreasing amplitude
        arg = (np.argsort(Amp))[::-1]
        if ( verbose):
            print ("Start time [sec] = %s") % np.array_str(Time[locin[arg]])
            print ("End time [sec] = %s") % np.array_str(Time[locout[arg]])
            print ("Duration [sec]: %s") % np.array_str(Time[locout[arg]]-Time[locin[arg]])
	if (plot):
		plt.figure(100)
		for i in range(njumps):
			plt.plot([Time[locin[i]],Time[locin[i]]],[0.,2.],'r') 
    	                plt.plot([Time[locout[i]],Time[locout[i]]],[0.,2.],'b')
		plt.axis(ymin=1. - 3.*threff , ymax = 1. + 3.*threff ) 

        return Time[locin[arg]],Time[locout[arg]],Amp[arg]
    return np.array([]),np.array([]),np.array([])

def correct_jumps(Time,Flux,StartTime,EndTime,WinSize, verbose = False  , dt = 32.):
    n = len(StartTime)
    for i in range(n):
     t1,t2 = StartTime[i],EndTime[i]
     before = np.where( (Time < t1-dt) & (Time >= t1-dt - WinSize) )[0]
     after = np.where( (Time > t2+dt) & (Time <= t2+dt + WinSize) )[0]
     if ( (len(after) > 2) & (len(before) > 2) ):
         fitfunc = lambda x, a,b,c : a + b*x # + c*x**2
         pb , pcov = curve_fit(fitfunc, Time[before], Flux[before])
         pa , pcov = curve_fit(fitfunc, Time[after], Flux[after])
         sf = (pb[1]*t2 + pb[0]) / (pa[1]*t2 + pa[0])
#         sf = (pb[2]*t2**2 + pb[1]*t2 + pb[0]) / (pa[2]*t2**2 + pa[1]*t2 + pa[0])
         if (verbose):
             print ("Start time= %f , end time= %f") % (t1,t2)
             print ("scaling factor= %f") %(sf)
         after = np.where( Time > t2)[0]
         Flux[after] = Flux[after]*sf
    return Flux

def fill_big_gaps(Flag,Time,Flux,Status,GapSizeMin,GapSizeMax,Sampling,NSampling,verbose = False):
    TimeShift = (Time[1:]-Time[0:-1])
    TimeShiftDate = (Time[1:]+Time[0:-1]) /2.
    MSampling = (Sampling[1:]+Sampling[0:-1]) /2.
    MeanSampling = np.mean(MSampling)
    if(verbose):
	print ("minimal time shift: %f") % (np.min(TimeShift))
        print ("maximal time shift %f") % (np.max(TimeShift))
        print ("mean    time shift: %f") % (np.mean(TimeShift))


    Gaps = (TimeShift >= GapSizeMin*MSampling ) & ( TimeShift <= GapSizeMax)
    ngaps = np.sum(Gaps)

    n = len(Time)
    if(ngaps>0):
        idx = np.where(Gaps)[0]
        if (verbose):
            print ("%d gaps detected with duration >%f (s) and  <%f (s)") % (ngaps,GapSizeMin*MeanSampling,GapSizeMax)
            print ("median gap duration =  %f") % (np.median(TimeShift[idx]))
            print ("shortest =  %f") %  (np.min(TimeShift[idx]))
            print ("longest =  %f") % (np.max(TimeShift[idx]))
        TimeNew = Time[0:idx[0]+1]
        FlagNew = Flag[0:idx[0]+1]
        FluxNew = Flux[0:idx[0]+1]
	SamplingNew = Sampling[0:idx[0]+1]
        StatusNew = Status[0:idx[0]+1]
        for j in range(ngaps):
            N = int (  (Time[idx[j]+1] - Time[idx[j]])/NSampling ) - 1
            flag = np.zeros(N,dtype=np.bool)
            flag[:] = False
            TimeNew = np.append(TimeNew,np.zeros(N)*NSampling+Time[idx[j]])
            FlagNew = np.append(FlagNew,flag)
            FluxNew = np.append(FluxNew,np.zeros(N))
            StatusNew = np.append(StatusNew,np.zeros(N,dtype=np.int32)*(2**30))
            SamplingNew = np.append(SamplingNew,np.zeros(N)+NSampling)
            if (j<ngaps-1):
                TimeNew = np.append(TimeNew,Time[idx[j]+1:idx[j+1]+1])
                FlagNew = np.append(FlagNew,Flag[idx[j]+1:idx[j+1]+1])
                FluxNew = np.append(FluxNew,Flux[idx[j]+1:idx[j+1]+1])
                StatusNew = np.append(StatusNew,Status[idx[j]+1:idx[j+1]+1])
		SamplingNew = np.append(SamplingNew,Sampling[idx[j]+1:idx[j+1]+1])
            else:
                TimeNew = np.append(TimeNew,Time[idx[j]+1:])
                FlagNew = np.append(FlagNew,Flag[idx[j]+1:])
                FluxNew = np.append(FluxNew,Flux[idx[j]+1:])
                StatusNew = np.append(StatusNew,Status[idx[j]+1:])
		SamplingNew = np.append(SamplingNew,Sampling[idx[j]+1:])
        return (FlagNew,TimeNew,FluxNew,StatusNew,SamplingNew)
    else:
        return (Flag,Time,Flux,Status,Sampling)
        

def fill_flagged_holes(SFluxFlag,SFlux,Flag,Time,Flux, verbose = False):

    bad = (  Flag == False )
    good =  ( SFluxFlag == True   )
#    good =  (  Flag == True )

    n =(bad).sum()
    if (verbose):
        print ("Number of filled holes: %d (/ %d, frac.: %f %%)") % (n,len(Flux),float(n)/float(len(Flux))*100.)
    if ( (n > 0) & ((good).sum() > 0) ):
#        Flux[bad] = np.interp(Time[bad],Time[good],Flux[good])
        Flux[bad] = np.interp(Time[bad],Time[good],SFlux[good])
        Flag[bad] = True

    return Flux,Flag
 
def flag_outliers(Flag,Flux,SFluxFlag,SFlux,StdDev,NSIGMA,verbose = False):

    goodidx =  np.where(  Flag & SFluxFlag )[0]
    if ( len(goodidx) > 3 ):
	diff =  np.abs( Flux[goodidx] - SFlux[goodidx] )
        outliers = np.where( diff  >  NSIGMA*StdDev[goodidx] ) [0]
        noutliers = len(outliers)
        if (verbose):
            print ("Number of detected outliers: %d") % (noutliers)
        if (noutliers>0):
#		print diff[outliers],(StdDev[goodidx])[outliers]
		Flag[goodidx[outliers]] = False
    return Flag

def bin_lc(array,flag,n):
	nr=int(float(array.shape[0])/float(n))
	array0 = np.reshape(array,(n,nr))
	flag0 = np.reshape(flag,(n,nr))
	arrayn = np.zeros(n)
	ncum = np.zeros(n,dtype=np.int)
	for j in range(nr):
		arrayn[:] = arrayn[:] + array0[:,j] * (flag0[:,j] == True ) 
	ncum[:] = np.sum( flag0[:,:] == True , 1)
	return arrayn,ncum

def rebin_status(status,flag,n):
        nr=int(float(status.shape[0])/float(n))
	status0 = np.reshape(status,(n,nr))
	flag0 = np.reshape(flag,(n,nr))
	statusn = np.zeros(n,dtype=np.int32)
	for j in range(nr):
		statusn[:] = statusn[:] | ( status0[:,j] * (flag0[:,j]==True) )
	return statusn



def translate_time_exo_lc(time,sampling):
	i32 = np.abs(sampling-32.) < 1e-10
        n32 = np.sum(i32)
        i512 = (i32 == False)
        time[i32] = time[i32] - (( 32  - 0.3081)/2. )/(3600.*24.)
        time[i512] = time[i512] +  ( - (32  - 0.3081)/2.  + 240.    ) /(3600.*24.)
	return time


def fdiff(x):
    '''
    Difference of a vector betweeen consecutive points
    '''
    d = np.zeros(len(x))
    d[:-1] = x[1:] - x[:-1]
    d[-1] = d[-2]
    return d

def valid_status(status,typeFile,DataVersion):
        # we flag the valid measurements
        if ( typeFile == 1): #  sismo channel
             flag = (status==0) |   ( status == 4)
        elif ( (typeFile >=2) & (typeFile<=4)  ): # exo channel
            #             flag =  (status==0) |   ( status == 32)
            if( DataVersion < 4.0):
                flag = ( checkbit(status,0) | checkbit(status,1) | checkbit(status,2) |  checkbit(status,7) |  checkbit(status,8) | checkbit(status,10)  |  checkbit(status,11)  ) == False
            else:
                flag = (status < 2**32) # in practice we take all the measurements

        elif ( (typeFile == 7) | (typeFile ==8) ): # kepler data
             flag = ( status < 2**15 ) # see Kepler Archive Manual, Thompson & Fraquelli, June 2015
	return flag

def downsampling_exo_lc(time,flux,status,flag,sampling,verbose=False):
	i32 = np.abs(sampling-32.) < 1e-10
	n32 = np.sum(i32)
	if (n32 > 0):
		## Detection of chunks of same sampling
    		dtexp = fdiff(sampling) # -480 = 512->32 s, 480 = 32->512s 480 marks the last of the sampling
    		sampchange = ( np.abs( np.abs(dtexp)- 480. ) < 1e-10)
    		nchunks = np.sum(sampchange) + 1
		if(verbose):
			print ('number of sequence with different sampling: %d') % (nchunks)
			print ('n 32s = %d , n 512s = %d ') % (n32,time.size-n32)
    		for chunk in range(nchunks):
		# chunk selection
        		if chunk == 0:
            		# first chunk
           			istart = 0 
        		else:
            			istart = (np.argwhere(sampchange)[chunk-1.])+1
	        	if chunk == nchunks-1:
            			# last chunk
            			iend = len(time) 
        		else:
            			iend = (np.argwhere(sampchange)[chunk])+1

        		# data for chunk
        		time0 = time[istart:iend]
			flux0 = flux[istart:iend]
			status0 = status[istart:iend]
			sampling0 = sampling[istart:iend]
			flag0 = flag[istart:iend]
			n = time0.size
			i32 = np.abs(sampling0-32.) < 1e-10
			n32 = np.sum(i32)
#			print 'n32=',n32
			if (n32 >0 ):
				p = int(float(n32)/16.)
				k = p*16
#				print 'p,k=',p,k
				if( k> 0):
					flux0,ncum = bin_lc(flux0[0:k],flag0[0:k],p)
					time00,tmp = bin_lc(time0[0:k],np.ones(k,dtype=np.bool),p)
					time0,ncum = bin_lc(time0[0:k],flag0[0:k],p)
					status0 = rebin_status(status0[0:k],flag0[0:k],p)
					flag0 = (ncum>7)
					flux0[ncum>0] = flux0[ncum>0] / ncum[ncum>0]
					time0[ncum>0] = time0[ncum>0] / ncum[ncum>0]
					time0[ncum==0] = time00[ncum==0]/16.
					status0 = status0 | (2**8 *(flag0==False))
				else:
					flux0 = np.array([])
					time0 = np.array([])
					status0 = np.array([],dtype=np.int32)
					flag0 = np.array([],dtype=np.bool)
			n = time0.size
			sampling0 = np.zeros(n)  + 512.
			if( chunk == 0):
				time1 = np.array(time0)
				flux1 = np.array(flux0)
				status1 = np.array(status0)
				sampling1 =  np.array(sampling0)
				flag1 = np.array(flag0)
			else:
				time1 = np.append(time1,time0)
				flux1 = np.append(flux1,flux0)
				status1 = np.append(status1,status0)
				sampling1 = np.append(sampling1,sampling0)
				flag1 = np.append(flag1,flag0)
		return time1,flux1,status1,flag1,sampling1			
	else:

		return time,flux,status,flag,sampling

class Record:
 def __init__(self,config=None):


     #-----------------------------------------------------------------------
 
     #Configuration de l'application à partir de config.xml
 
     #----------------------------------------------------------------------
     if (config==None):
        self.config=None
        try:
            doc=minidom.parse('../config.xml')
        except:
            doc=minidom.parse('config.xml')
     else:
        try:
            doc=minidom.parse(config)
            self.config=config
        except:
            traceback.print_exc()
            print('Incorrect file {}'.format(config))

     elem= doc.getElementsByTagName('fits_input')
     self.fits_input= elem.item(0).firstChild.nodeValue

     elem= doc.getElementsByTagName('fits_output')
     self.fits_output= elem.item(0).firstChild.nodeValue

     elem= doc.getElementsByTagName('pp_output')
     self.pp_output= elem.item(0).firstChild.nodeValue

     elem= doc.getElementsByTagName('final_output')
     self.final_output= elem.item(0).firstChild.nodeValue

     self.Version= ""
     self.NVersion= ""     
     self.PPVersion= ""



     if not os.path.exists(self.fits_output):  
        os.makedirs(fits_output)


    #-----------------------------------------------------------------------
   
     self.Origin=int(-1)
     self.StarID= int (-1)
     self.Type=""
     self.StarName=""
     self.Teff=float(-1)
     self.Grav=float(-1)
     self.Mag_v=float(-1)
     self.Mag_r=float(-1)
     self.Alpha=float(-1)
     self.Delta=float(-1)
     self.SpecType=""
     self.SubClass=""
     self.LumClass=""
     self.ColDef=float(-1)
     self.Flux=list()
     self.Time=list()
     self.Freq=list()
     self.PSD=list()
     self.Flags=list()
     self.Indices=list()
     self.files=list()
     self.Sampling=list()
     self.DutyCycle=float(-1)
    


 def __getattr__(self, nom):

         print("Alerte ! Il n'y a pas d'attribut {} !!!!!!!!!!!!!!!!!!!".format(nom))

 
 def preprocessing(self,param,verbose = False , plot = False):

     CorrJumps = param['CorrJumps']  # Turn on/off the jump correction
     ThresholdJumps = param['ThresholdJumps'] # Threshold above which a (relative) variation is considered as a jump 
     MaxJumps = param['MaxJumps'] # Maximum number of jumps removed 
     NIterJumps = param['NIterJumps'] # Number of iterations performed during the job correction

     CorrIndivTrend = param['CorrIndivTrend'] # Turn on/off the trend correction for each individual light-curve
     CorrGlobalTrend = param['CorrGlobalTrend'] # Turn on/off the trend correction for the complete ligth-curve 

     FillBigGaps = param['FillBigGaps'] # Fill the big gaps (larger than GapSizeMin and shorter than GapSizeMax)
     FillHoles = param['FillHoles'] # Fill the holes (included the big gaps if FillBigGaps=True)
     GapSizeMin = param['GapSizeMin'] # Minimum gap duration [in sampling time unit]
     GapSizeMax = param['GapSizeMax'] # Maximum gap duration [in sec]

     WinSizeSmoothFill = param['WinSizeSmoothFill'] # Size of moving window used for filling the holes [in sec]
     WinSizeSmoothMatch = param['WinSizeSmoothMatch'] # Size of moving window used for matching the individual light-curves  [in sec]

     NSigma =  param['NSigma']  # Parameter used for the outliers detection
     JoinLC =  param['JoinLC']  # Join together indidual Light-Curves
     DataVersion = param['DataVersion'] # version of the data product
     
     def preprocess_file(time,flux,status,typeFile,sampling,NSampling,DataVersion):
        # we flag the valid measurements
        flag = valid_status(status,typeFile,DataVersion)
        n = np.sum(flag)
        if (n<10):
            print ("they are too few measurements (%d), we skip this file") % (n)
            return (False,0,0,0,0,0)
        if (verbose):
            print ("number of valid data = %d (frac: %f %%)") % (n,float(n)/float(len(flag))*100. )
	# we compute the smoothed light-curve
        sflux,  sfluxflag  , stddev = smooth_lc(time,flux,flag,WinSizeSmoothFill,typeFile,sampling,NSampling,verbose=verbose,type=1)
	flag = flag_outliers(flag,flux,sfluxflag,sflux,stddev,NSigma,verbose = verbose)
	if ( (typeFile >=2) & (typeFile<=3) ): # exo channel
		time,flux,status,flag,sampling= downsampling_exo_lc(time,flux,status,flag,sampling,verbose=verbose)
		NSampling = float(np.median(sampling)) # nominal sampling time (sec)
		if(verbose):
			print ('new nominal sampling: %d') % (NSampling)
		sflux,  sfluxflag  , stddev = smooth_lc(time,flux,flag,WinSizeSmoothFill,typeFile,sampling,NSampling,verbose=verbose,type=1)
		flag = flag_outliers(flag,flux,sfluxflag,sflux,stddev,NSigma,verbose = verbose)
        if (CorrJumps & ( (typeFile >=2) & (typeFile<=4) ) ): # exo channel
	    '''
            it = 0
            njumps = MaxJumps-1
            while ( (it < NIterJumps) & (njumps>0)& (njumps<MaxJumps) ):
                # we compute the smoothed light-curve
                valididx = np.where(sfluxflag)[0]

                if ( len(valididx) < 2):
                    return (False,0,0,0,0)

                StartTime,EndTime,Amp = detect_jumps(time[valididx],sflux[valididx],stddev[valididx],WinSizeSmooth,NSampling,ThresholdJumps,verbose=verbose,plot=plot)
                njumps = len(StartTime)
                if ( njumps > MaxJumps):
                    print ("Warning ! too many jumps detected (%d>%d), we correct the first %d jumps ") % (njumps,MaxJumps,MaxJumps)
                    flux[valididx] = correct_jumps(time[valididx],flux[valididx],StartTime[0:MaxJumps],EndTime[0:MaxJumps],WinSizeSmooth,verbose=verbose,dt=NSampling)
                elif (njumps>0):
                    flux[valididx] = correct_jumps(time[valididx],flux[valididx],StartTime,EndTime,WinSizeSmooth,verbose=verbose,dt=NSampling)
                it = it+1
	    	sflux,  sfluxflag  , stddev  = smooth_lc(time,flux,flag,WinSizeSmooth,typeFile,sampling,NSampling,verbose=verbose,type=1)
	    '''
            time,flux,status,njumps= JumpFilter.JumpFilter(time,flux,sampling,status,flag,self.StarID,ephemfile='',verbose=verbose,outplot=False)
	    if(njumps>0):
		    sflux,  sfluxflag  , stddev = smooth_lc(time,flux,flag,WinSizeSmoothFill,typeFile,sampling,NSampling,verbose=verbose,type=1)
		    flag = flag_outliers(flag,flux,sfluxflag,sflux,stddev,NSigma,verbose = verbose)

        if (FillBigGaps):
            flag,time,flux,status,sampling = fill_big_gaps(flag,time,flux,status,GapSizeMin,GapSizeMax,sampling,NSampling ,verbose = verbose)

        sflux,sfluxflag  = smooth_lc(time,flux,flag,WinSizeSmoothFill,typeFile,sampling,NSampling,verbose=verbose,type=2)
        sfluxMatch,sfluxflagMatch  = smooth_lc(time,flux,flag,WinSizeSmoothMatch,typeFile,sampling,NSampling,verbose=verbose,type=2)

        return (True,time,flux,sflux,status,sampling,flag,sfluxflag,sfluxMatch,sfluxflagMatch)
    

     ## if( self.Origin != 1 ):
     ##     print "can only manage for the moment COROT ligth-curves, we skip this star"
     ##     return False
     
     Time = np.array([])
     Flux = np.array([])
     SFlux = np.array([])
     Sampling = np.array([])
     Status = np.array([],dtype=np.int32)
     Flag = np.array([],dtype=np.bool)
     SFluxFlag = np.array([],dtype=np.bool)
     fitfunc = lambda x, a,b: a*x+b
     i=0
     if (plot):
         import matplotlib.pyplot as plt
         plt.figure(1)
         plt.clf()
         plt.figure(2)
         plt.clf()
         plt.show(block=False)
     dutytime = 0.
     for f in self.files:
         if (verbose):
             print ('processing %s') % (f.FileName)
         status = np.int32(f.Flags)
         time=np.float64(f.Time)
         flux=np.float64(f.Flux)
         typeFile = GF.isFile_Star(f.FileName)
	 flag = valid_status(status,typeFile,DataVersion)
         if ( (typeFile >=2) & (typeFile<=3) ): # exo channel
             sampling = f.Sampling
#	     time = translate_time_exo_lc(time,sampling)
#	     time,flux,status,flag,sampling= downsampling_exo_lc(time,flux,status,flag,sampling,verbose=verbose)
	     NSampling = float(np.median(sampling)) # nominal sampling time (sec)
         elif ( (typeFile <1) | ( typeFile > 8)):
             print ("We do not handle file of type = %d, we skip this file") % (typeFile)
             continue
         else:
             NSampling = f.Sampling  # nominal sampling time (sec)
             sampling = np.zeros(f.Time.size)
             sampling[:] = f.Sampling
         if (verbose):
             print ("nominal sampling: %f [s]") % (NSampling)
             print ("minimal sampling: %f") % (np.min(sampling))
             print ("maximal sampling: %f") % (np.max(sampling))
         n = np.sum(flag)
         if (n<10):
             print ("they are too few valid measurements (%d<10) in file %s, we skip this file") % (n,f.FileName)
             continue
         if (i==0):
             self.Time0 = time[0]
         time = (time-self.Time0)*86400.  # time vector in seconds
         if ( JoinLC & (i>0)):
                time = (time-time[0]) + Time[-1] + NSampling
         popt, pcov = curve_fit(fitfunc, time[flag], flux[flag] )
         if (i==0):
             self.Flux0 = (popt[0]*time[0]+popt[1])
         if (CorrIndivTrend):
             flux = flux / (popt[0]*time+popt[1])  # we remove the trend for this light-curve
         else:
             flux = flux / self.Flux0
         if (plot):
             plt.figure(1)
             plt.plot(time[flag],flux[flag])
             plt.show(block=False)
             
         valid,time,flux,sflux,status,sampling,flag,sfluxflag,sfluxmatch,sfluxflagmatch = preprocess_file(time,flux,status,typeFile,sampling,NSampling,DataVersion)
         if( i==0):
             flux1 = (sfluxmatch[sfluxflagmatch])[-1]                
         if (i>0): # we match the flux with that of the previous ligth-curve
#             flux = flux *  ( (poptp[0]*time[0]+poptp[1]  ) / (popt[0]*time[0]+popt[1]  ) )
             fluxp = flux1
             ratio = ( fluxp / (sfluxmatch[sfluxflagmatch]) [0] )
             flux = flux *  ratio
             sflux = sflux * ratio
             sfluxmatch = sfluxmatch*ratio
             flux1 = (sfluxmatch[sfluxflagmatch])[-1]

         if (valid):
            Flux=np.append(Flux,flux)
            SFlux=np.append(SFlux,sflux)
            Time=np.append(Time,time)
	    Sampling=np.append(Sampling,sampling)
            Status=np.append(Status,status)
            Flag=np.append(Flag,flag)
            SFluxFlag=np.append(SFluxFlag,sfluxflag)
            i += 1
            if (plot):
                plt.figure(2)
                plt.plot(time[flag],flux[flag])
                plt.show(block=False)

     if (i ==0):
         print "No valid files, we skip this star"
         return False
     if (verbose):
         print ("number of valid files: %d") % (i)
     if (CorrGlobalTrend):
         popt, pcov = curve_fit(fitfunc, Time[Flag], Flux[Flag] )
         Flux[Flag] = Flux[Flag] / (popt[0]*Time[Flag]+popt[1])
         SFlux[Flag] = SFlux[Flag]  / (popt[0]*Time[Flag]+popt[1])
     if (plot):
     	plt.figure(3)
     	plt.clf()
        plt.plot( Time[SFluxFlag], Flux[SFluxFlag] )
        plt.plot(  Time[SFluxFlag], SFlux[SFluxFlag] )
     if (FillHoles):
         ##	 SFlux,  SFluxFlag   = smooth_corot_exo_lc(Time,Flux,Flag,WinSizeSmooth, verbose = verbose , type=2)
         if (plot):
             FlagOld = Flag.copy()
         Flux,Flag = fill_flagged_holes(SFluxFlag,SFlux,Flag,Time,Flux, verbose=verbose)
         if (plot):
            plt.figure(4)
            plt.clf()
            plt.plot( Time, Flux )
            plt.plot(  Time[SFluxFlag], SFlux[SFluxFlag] )
            plt.plot( Time[FlagOld == False] ,  Flux[FlagOld == False], 'r+')

     self.Time = Time[Flag]
     self.Flux = Flux[Flag] -1.
     self.Flags = Status[Flag]
     self.Sampling = Sampling[Flag]
     dutytime=0.
     if( Flag.sum() >0):
	dutytime = self.Sampling.sum()
     else:
         print "No valid data, we skip this star"
         return False
     if(dutytime > 0.):
     	self.DutyCycle = dutytime/ (self.Time[-1]  + self.Sampling[-1] - self.Time[0])
	if(verbose):
		print 'Duty-cycle: %f' % (self.DutyCycle)
		
     # PSD computed using Lomb-scargle periodogram by B. Leroy code (2012, Astron. Astrophys. 545, A50)
     (self.Freq,self.PSD) = pynfftls.period(self.Time,self.Flux,1.,1.)   #f:Frequency vector ; psd: Power Spectrum Density vector
     # normalisation of the PSD such that its integral from nu=0 to the Nyquist frequency is equal to the variance ("one-sided power spectrum")
     N = float(len(self.PSD))
     self.PSD = self.PSD* np.var(self.Flux, ddof=1)/self.Freq[0]/N*1e6	#  Unit: ppm^2/muHz ; ddof: Delta Degrees of Freedom (N-ddof)
     if (plot):
         plt.figure(5)
         plt.clf()
         plt.plot(self.Freq,self.PSD)
         plt.plot(self.Freq,smooth(self.PSD,101))
         plt.loglog()
         plt.show(block=False)
     return True

 def set_Origin(self,SOrigin):

  if SOrigin.lower().strip()=="corot":
     self.Origin=1
  elif SOrigin.lower().strip()=="kepler":
    self.Origin=2
  elif SOrigin.lower().strip()=="oggle":
    self.Origin=3
  else:
    self.Origin=0

 def FileDone(self,f):
   i=0
   for fi in self.files:
     if f.LCStart<fi.LCStart:
         return i
     else:
      i=i+1
      if fi.FileName == f.FileName:
         return -1
   return i
 
 def New_ver(self):
    return self.NVersion
 
 def update_ver(self):
    self.Version=self.NVersion 
   
 
 def read_bpp(self,current,origin):
   file_name=self.fits_output+"/Data_{}--{}.bin".format(int(origin),int(current))
   with open(file_name, 'rb') as fichier:
                    mon_depickler = pickle.Unpickler(fichier)
                    l = mon_depickler.load()
		    fichier.close()

   f=Files(None)
   self.Origin=l['Origin']
   self.StarID=l['StarID']
   self.StarName=l['StarName']
   self.Type=l['Type']
   self.Version=l['Version']
   self.Teff=l['Teff']
   self.Grav=l['Grav']
   self.Alpha=l['Alpha']
   self.Delta=l['Delta']
   self.Mag_v=l['Mag_v']
   self.Mag_r=l['Mag_r']
   self.SpecType=l['SpecType']
   self.SubClass=l['SubClass']
   self.LumClass=l['LumClass']
   self.ColDef=l['ColDef']
   self.files=[]
   i=0
   stop=(len(l)-15)/10
   while i<stop:

       
       f=Files(None)
       f.FileName=l['FileName{}'.format(i)]
       f.Type=l['TypeFile{}'.format(i)]
       f.FCD=l['FCD{}'.format(i)]
       f.LCStart=l['LCStar{}'.format(i)]
       f.LCEnd=l['LCEnd{}'.format(i)]
       f.Flux=l['Flux{}'.format(i)]
       f.Time=l['Time{}'.format(i)]
       f.Flags=l['Flags{}'.format(i)]
       f.Run=l['Run{}'.format(i)]
       f.Sampling=l['Sampling{}'.format(i)]
       i=i+1
       self.files.append(f)

        
 
 def write_bpp(self,current,origin,version):
   file_name=self.fits_output+"/Data_{}--{}.bin".format(int(origin),int(current))
   l={}
   l['Origin']=self.Origin
   l['StarID']=self.StarID
   l['StarName']=self.StarName
   l['Type']=self.Type
   l['Version']=version
   l['Teff']=self.Teff
   l['Grav']=self.Grav
   l['Alpha']=self.Alpha
   l['Delta']=self.Delta
   l['Mag_v']=self.Mag_v
   l['Mag_r']=self.Mag_r
   l['SpecType']=self.SpecType
   l['SubClass']=self.SubClass
   l['LumClass']=self.LumClass
   l['ColDef']=self.ColDef
   i=0
   for f in self.files:
       l['FileName{}'.format(i)]=f.FileName
       l['TypeFile{}'.format(i)]=f.Type
       l['FCD{}'.format(i)]=f.FCD
       l['LCStar{}'.format(i)]=f.LCStart
       l['LCEnd{}'.format(i)]=f.LCEnd
       l['Flux{}'.format(i)]=f.Flux
       l['Time{}'.format(i)]=f.Time
       l['Flags{}'.format(i)]=f.Flags
       l['Run{}'.format(i)]=f.Run
       l['Sampling{}'.format(i)]=f.Sampling
       i=i+1
   
   with open(file_name, 'wb') as fichier:
                 mon_pickler = pickle.Pickler(fichier)
                 mon_pickler.dump(l)
		 fichier.close()

 def read_app(self,current,origin):
   file_name=self.pp_output+"/PPData_{}--{}.bin".format(int(origin),int(current))
   with open(file_name, 'rb') as fichier:
                    mon_depickler = pickle.Unpickler(fichier)
                    l = mon_depickler.load()
		    fichier.close()

   file=Files(None)
   self.Origin=l['Origin']
   self.StarID=l['StarID']
   self.StarName=l['StarName']
   self.Type=l['Type']
   self.Version=l['Version']
   self.PPVersion=l['PPVersion']
   self.Teff=l['Teff']
   self.Grav=l['Grav']
   self.Alpha=l['Alpha']
   self.Delta=l['Delta']
   self.Mag_v=l['Mag_v']
   self.Mag_r=l['Mag_r']
   self.SpecType=l['SpecType']
   self.SubClass=l['SubClass']
   self.LumClass=l['LumClass']
   self.ColDef=l['ColDef']
   self.Flux=l['Flux']
   self.Time=l['Time']
   self.Freq=l['Freq']
   self.PSD=l['PSD']
   self.Flags=l['Flags']
   self.Sampling=l['Sampling']
   self.DutyCycle=l['DutyCycle']
   self.files=[]
   size=len(l.keys())
   i=0
   stop=(len(l)-21)/6
   while i<stop:
       file=Files(None)
       file.FileName=l['FileName{}'.format(i)]
       file.Type=l['TypeFile{}'.format(i)]
       file.FCD=l['FCD{}'.format(i)]
       file.LCStart=l['LCStar{}'.format(i)]
       file.LCEnd=l['LCEnd{}'.format(i)]
       file.Run=l['Run{}'.format(i)]
       i=i+1
       self.files.append(file)


 def write_app(self,current,origin,PPVersion):
   file_name=self.pp_output+"/PPData_{}--{}.bin".format(int(origin),int(current))
   l={}
   l['Origin']=self.Origin
   l['StarID']=self.StarID
   l['StarName']=self.StarName
   l['Type']=self.Type
   l['Version']=self.Version
   l['PPVersion']=PPVersion
   l['Teff']=self.Teff
   l['Grav']=self.Grav
   l['Alpha']=self.Alpha
   l['Delta']=self.Delta
   l['Mag_v']=self.Mag_v
   l['Mag_r']=self.Mag_r
   l['SpecType']=self.SpecType
   l['SubClass']=self.SubClass
   l['LumClass']=self.LumClass
   l['ColDef']=self.ColDef
   l['Flux']=self.Flux
   l['Time']=self.Time
   l['Flags']=self.Flags
   l['Freq']=self.Freq
   l['PSD']=self.PSD
   l['Sampling']=self.Sampling
   l['DutyCycle']=self.DutyCycle
   i=0
   
   for file in self.files:
       l['FileName{}'.format(i)]=file.FileName
       l['TypeFile{}'.format(i)]=file.Type
       l['FCD{}'.format(i)]=file.FCD
       l['LCStar{}'.format(i)]=file.LCStart
       l['LCEnd{}'.format(i)]=file.LCEnd
       l['Run{}'.format(i)]=file.Run
       i=i+1
   
   with open(file_name, 'wb') as fichier:
                 mon_pickler = pickle.Pickler(fichier)
                 mon_pickler.dump(l)
		 fichier.close()
                 
 def read(self,current,origin):
   file_name=self.final_output+"/FData_{}--{}.bin".format(int(origin),int(current))
   with open(file_name, 'rb') as fichier:
                    mon_depickler = pickle.Unpickler(fichier)
                    l = mon_depickler.load()
		    fichier.close()

   file=Files(None)

   self.Origin=l['Origin']
   self.StarID=l['StarID']
   self.StarName=l['StarName']
   self.Type=l['Type']
   self.Version=l['Version']
   self.PPVersion=l['PPVersion']
   self.Teff=l['Teff']
   self.Grav=l['Grav']
   self.Alpha=l['Alpha']
   self.Delta=l['Delta']
   self.Mag_v=l['Mag_v']
   self.Mag_r=l['Mag_r']
   self.SpecType=l['SpecType']
   self.SubClass=l['SubClass']
   self.LumClass=l['LumClass']
   self.ColDef=l['ColDef']
   self.files=[]
   self.Indices=[]
   size=len(l.keys())
   i=0
   while True:
     try:
       file=Files(None)
       file.FileName=l['FileName{}'.format(i)]
       file.Type=l['TypeFile{}'.format(i)]
       file.FCD=l['FCD{}'.format(i)]
       file.LCStart=l['LCStar{}'.format(i)]
       file.LCEnd=l['LCEnd{}'.format(i)]
       file.Run=l['Run{}'.format(i)]
       i=i+1
       self.files.append(file)
     except:
         break
   i=0
   while True:
     try:
       indice=Indice(self.config)
       indice.Value=l['Value{}'.format(i)]
       indice.Type=l['TypeInd{}'.format(i)]
       indice.PrecP=l['PrecP{}'.format(i)]
       indice.PrecN=l['PrecN{}'.format(i)]
       indice.AccP=l['AccP{}'.format(i)]
       indice.AccN=l['AccN{}'.format(i)]
       indice.Asym_prec=l['Asym_prec{}'.format(i)]
       indice.Asym_acc=l['Asym_acc{}'.format(i)]
       indice.Date=l['Date{}'.format(i)]
       indice.Version=l['Version{}'.format(i)]
       indice.Comment=l['Comment{}'.format(i)]
       indice.Status=l['Status{}'.format(i)]
       i=i+1
       self.Indices.append(indice)
     except:
         break
         
 def write_app_fits(self,current,origin):
#	file_name=self.fits_input+ '/'+ self.files[0].FileName
#	hdulist = pyfits.open(file_name)
        file_name=self.pp_output+"/{}.fits".format(int(current))
#        file_name="/data/samadi/ssi/data/corot/LRc01_EN2_V3_2_REZA/"+self.files[0].FileName
        col11 = pyfits.Column(name='Time',format='D',array=(self.Time/86400.)+self.Time0,unit='Julian Day')        
        col12 = pyfits.Column(name='Flux',format='D',array=(self.Flux+1.)*self.Flux0)
	tbhdu1 = pyfits.new_table(pyfits.ColDefs([col11,col12]))
        col21 = pyfits.Column(name='Frequency',format='D',array=self.Freq,unit='Hz')        
        col22 = pyfits.Column(name='PSD',format='D',array=self.PSD,unit='ppm2/muHz')
        tbhdu2 = pyfits.new_table(pyfits.ColDefs([col21,col22]))
#	hdu = pyfits.PrimaryHDU(None,hdulist[0].header)
	hdu = pyfits.PrimaryHDU()
	thdulist = pyfits.HDUList([hdu,tbhdu1,tbhdu2])
#	thdulist = pyfits.HDUList([hdu,tbhdu1])
        print 'saving ',file_name
	thdulist.writeto(file_name)
		
 def write(self,current,origin):
   file_name=self.final_output+"/FData_{}--{}.bin".format(int(origin),int(current))
   l={}
   l['Origin']=self.Origin
   l['StarID']=self.StarID
   l['StarName']=self.StarName
   l['Type']=self.Type
   l['Version']=self.Version
   l['PPVersion']=self.PPVersion
   l['Teff']=self.Teff
   l['Grav']=self.Grav
   l['Alpha']=self.Alpha
   l['Delta']=self.Delta
   l['Mag_v']=self.Mag_v
   l['Mag_r']=self.Mag_r
   l['SpecType']=self.SpecType
   l['SubClass']=self.SubClass
   l['LumClass']=self.LumClass
   l['ColDef']=self.ColDef

   i=0
   for file in self.files:
       l['FileName{}'.format(i)]=file.FileName
       l['TypeFile{}'.format(i)]=file.Type
       l['FCD{}'.format(i)]=file.FCD
       l['LCStar{}'.format(i)]=file.LCStart
       l['LCEnd{}'.format(i)]=file.LCEnd
       l['Run{}'.format(i)]=file.Run
       i=i+1

   i=0
   for indice in self.Indices:
       l['Value{}'.format(i)]=indice.Value
       l['TypeInd{}'.format(i)]=indice.Type
       l['PrecP{}'.format(i)]=indice.PrecP
       l['PrecN{}'.format(i)]=indice.PrecN
       l['AccP{}'.format(i)]=indice.AccP
       l['AccN{}'.format(i)]=indice.AccN
       l['Asym_prec{}'.format(i)]=indice.Asym_prec
       l['Asym_acc{}'.format(i)]=indice.Asym_acc
       l['Date{}'.format(i)]=indice.Date
       l['Version{}'.format(i)]=indice.Version
       l['Comment{}'.format(i)]=indice.Comment
       l['Status{}'.format(i)]=indice.Status
       i=i+1
    
   
   with open(file_name, 'wb') as fichier:
                 mon_pickler = pickle.Pickler(fichier)
                 mon_pickler.dump(l)
		 fichier.close()

if __name__ == '__main__':
    a=Record()
    a.write_bpp(221414234,2)
    #print ((a.read()))
    #b=input()
