'''
    Auto-correlation program 

	This file is part of the Stellar Seimic Indices pipeline

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

	Copyright (C) 2016 by R. Peralta 
'''

import numpy as np
from pylab import *
import matplotlib.pyplot as plt
import matplotlib.cm as cm
from os import chdir
import pynfftls
import autocorrel_func as acf
reload(acf)
from operator import itemgetter
from mygauss_fit import *
import fftw3
import Freq_orbitale as Forb
reload(Forb)


def autocorrelation_list (time,flux,f,psd,N_filter,nuo,c,N,nu_lim,N_auto,graph,passage,origin,StarID,fitnumax=False):


	""""""""""""""""""""""""""""" PSD COMPUTATION """""""""""""""""""""""""""""

	#nuo:      First frequency taken in [muHz]. Typically for Red Giant : nuo = 3 microHz 
	#c:        Common ratio for geometric progression. Typically c=sqrt(sqrt(2))
	#N_filter: Nomber of filters used for the autocorrelation. Typically N_filter = 11
	#N=3.:     delta nu multiplication factor. Typically N=3.
	#nu_lim:   The code search a numax and delta_nu until limit frequency nu_lim is reached. Typically nu_lim = 110	for CoRoT
	#N_auto:   Number of autocorrelation computation. Typically N_auto = 2	

	freq_nyquist = np.max(f) * 1e6
	y = np.roll(f,1)
	dify = f-y   
	freq_res = np.median(dify)*1e6		# Frequency resolution

	if (nu_lim == 'nyquist'): 
		nu_lim = freq_nyquist
		N_filter = int (round( (math.log(nu_lim)-math.log(nuo))/math.log(c) ) )  + 1


	if (origin ==1): # corot data
		p = Forb.Freq_orbitale (f,psd,StarID)			# Replace orbital frequencies by zeros
	else:
		p = psd

	if graph[0]==1:
		plt.figure(0)
		plt.clf()
		plt.xlabel('Time [s]')
		plt.ylabel('')
		plt.title('Light curve')
		plt.plot(time,flux)


	if graph[1]==1:
		if passage==1: 
		    plt.figure(1)
		    plt.clf()
		    plt.title('PSD')
		else: 
		    plt.figure(4)
		    plt.clf()
		    plt.title('PSD')		# plt.title('PSD without background') # si l'on retire le BG

		plt.xlabel('Frequency [muHz]')
		plt.ylabel('Power [ppm2/muHz]')
		plt.plot(f*1e6,p)
		#plt.loglog()

        
	n = len(f)


	""""""""""""""""""""""""""""" AUTOCORRELATION COMPUTATION """""""""""""""""""""""""""""

	#fftplan optimises the computational time 
	p_filt = np.zeros(2*n+1)			#input of fftplan
	p_ft = np.zeros(n+1,dtype=np.complex128)	#output of fftplan
	fftplan = fftw3.Plan(p_filt,p_ft, direction='forward', flags=['estimate']) #ou flags=['estimate']
	#for more details about the fftplan, see http://www.fftw.org/doc/Complex-One_002dDimensional-DFTs.html 
	#http://www.fftw.org/doc/One_002dDimensional-DFTs-of-Real-Data.html#One_002dDimensional-DFTs-of-Real-Data


	for j in range(0,N_auto):
	    if graph[2]==1:
		fig = plt.figure(j+5)
		if passage==2: fig = plt.figure(j+13)
		plt.clf()
		ax = fig.add_subplot(111)
		cm = plt.get_cmap('spectral')	#For different map colors, see: http://wiki.scipy.org/Cookbook/Matplotlib/Show_colormaps
		ax.set_color_cycle([cm(1.*i/N_filter) for i in range(N_filter)])

	    if graph[5]==1:
		fig = plt.figure(j+8)
		if passage==2: fig = plt.figure(j+16)
		plt.clf()
		ax = fig.add_subplot(111)
		cm = plt.get_cmap('spectral')	#For different map colors, see: http://wiki.scipy.org/Cookbook/Matplotlib/Show_colormaps
		ax.set_color_cycle([cm(1.*i/N_filter) for i in range(N_filter)])


	    #initialization
	    i=0
	    nu=zeros(N_filter)
	    delta_nu=zeros(N_filter)
	    maxi=zeros(N_filter)
	    tau=zeros(N_filter)
            nu[0] = nuo
            delta_nu[0] = 0.280 * nuo**0.747  	#Mosser et al 2010
            delta_nu_folded = 0.280 * (2.*freq_nyquist - nu[i]) **0.747


	    while (nu[i]<min(nu_lim,freq_nyquist-N/2.*delta_nu[i])) & (i<N_filter):	#nu<130 to avoid the frequency orbital; i<N_filter to respect the number of filter
		auto_time,auto_value,p_filtered,gamme, cos_filter=acf.autocorrel_func(N,nu[i],delta_nu[i],delta_nu_folded, f,p,fftplan,p_filt,p_ft)


		# --- If auto_value[i][gamme] is not empty (gamme != []), we have to check its shape, 
		#in order to determine its extremum and remove false maxima

		if gamme != []:

		    # Test if the autocorrelation begins by a decreasing slope 	    
		    k=1
		    neg1=[0]

		    while (auto_value[gamme[k-1]]>auto_value[gamme[k]]) & (k<len(gamme)-1):
			neg1.append(k)
			k+=1
				
			if k==len(gamme)-1:	#in case where all the shape of auto_value[i] is decreasing
			    neg1=[len(gamme)]
			    #print "the shape of the filter", nu[i], "is entirely decreasing"


		    # Test if the autocorrelation finishes by a increasing slope
		    k=len(gamme)-1
		    neg2=[len(gamme)]

		    while (neg1[-1]!=len(gamme)) & (auto_value[gamme[k]]>auto_value[gamme[k-1]]) & (k>0):
			neg2.append(k)
			k-=1
				
			if k==0:		#in case where all the shape of auto_value[i] is decreasing
			    neg2.append(k) 
			    #print "the shape of the filter", nu[i], "is entirely increasing"


		    # Determination of the autocorrelation maximum value and the corresponding time 'tau'
    		    if len(gamme[neg1[-1]:neg2[-1]]) > 1:  
			maxi[i] = max(auto_value[gamme[neg1[-1]:neg2[-1]]])
       			ind = np.argmax(auto_value[gamme[neg1[-1]:neg2[-1]]])
       			tau[i] = auto_time[gamme[ind]]

		    else:	# The shape of auto_value[i] is entirely decreasing or increasing
		   	maxi[i] = -1.
		   	tau[i] = -1.
		


		# --- If auto_time[i][gamme] is empty
		else:
		    #print "the filter", nu[i], "is empty"
		    maxi[i] = -1.
		    tau[i] = -1.

		if (origin == 1):
			# removing values close to the day : 11.57 muHz +/- 2 * internal uncertainty
			delta_nu_i = 2./tau[i]*1e6
			#error determination, cf equation (A.8) Mosser and Apourchaux (2009)
			beta = 0.763
			b = 5			#noise amplitude
			W_filter = N*delta_nu[i]	#filter width	
			err_deltanu = (beta*b*delta_nu_i**2)/(2*pi*maxi[i]*W_filter)
			if( (delta_nu_i < 11.574 + 1.)  & (delta_nu_i > 11.574 - 1.) ):
				maxi[i] = -1.
				tau[i] = -1.

		# --- plot
		if graph[2]==1:
		    color = cm(1.*i/N_filter)
		    plt.plot(auto_time[gamme[neg1[-1]:neg2[-1]]],auto_value[gamme[neg1[-1]:neg2[-1]]],label=("nu= %5.2f ") % (nu[i]))

		if graph[5]==1:
		    color = cm(1.*i/N_filter)
		    plt.plot(f*1e6,p_filtered)	#,label=("nu= %5.2f ") % (nu[i])

	    	i+=1

		if i==N_filter:
		   break

		elif (i<N_filter) & (j==0) & (N_auto >1):
		   nu[i] = c**i * nuo
		elif (i<N_filter) &  ( (j!=0) | (N_auto==1) ):
		   nu[i] = c*i + nuo 

		delta_nu[i] = 0.280 * nu[i]**0.747
                delta_nu_folded = 0.280 * (2.*freq_nyquist - nu[i]) **0.747

	    ind = np.argmax(maxi)
	    nuo=nu[ind]*2/3. # 0.8	#nuo is calculated in order to obtain the frequency with the heighest autocorrelation  
	    last=nu[ind]*4/3. # 1.2
	    #c=(last-nuo)/(N_filter-1)	#New arithmetic progression. the second scan have to be thinner
	    c=freq_res			#New arithmetic progression with a ratio equal to the resolution of the spectrum.
	    NFilter = int(round((last-nuo)/c)) + 1

	    

	#plot autocorrelation figure
	if graph[2]==1:
		for j in range(0,N_auto):  
		    plt.xlabel('Time [s]')
		    plt.ylabel('Auto correlation')
		    plt.legend()

		    if passage==1:
		        plt.figure(j+5)
			if j==0: plt.title('Autocorrelation curve - 1st calculation (all spectra)')
			else:    plt.title('Autocorrelation curve - 1st calculation (around numax (ACF))')

		    else:
			plt.figure(j+13)
			plt.title('Autocorrelation curve - 2nd calculation (around numax (fit))')

	if graph[5]==1:
		for j in range(0,N_auto):  
		    plt.xlabel('Frequency [muHz]')
		    plt.ylabel('Power [ppm2/muHz]')
		    #plt.legend()

		    if passage==1:
		        plt.figure(j+8)
			if j==0: plt.title('PSD divided by filters - 1st calculation')
			else:    plt.title('PSD divided by filters - 1st calculation (around numax (ACF))')

		    else:
			plt.figure(j+16)
			plt.title('PSD divided by filters - 2nd calculation (around numax (fit))')

	
	""""""""""""""""""""""""""""""""" DATA ANALYSIS """""""""""""""""""""""""""""""""
	"""
	"""
	
	rm=np.where(maxi==-1)[0]	#Removing bad points
	nu = np.delete(nu,rm)
	maxi = np.delete(maxi,rm)
	tau = np.delete(tau,rm)
	
	Amax = maxi.max()
	ind = np.argmax(maxi)
	

	""" ---Fit + Plot--- """

	""" Delta nu determination """
	#gaussian fit to determine delta_nu
	#time_fit2, fit2, popt2, pcov2 = mygauss_fit (tau,maxi,100000)

	deltanu = (2./tau[ind])*1e6	#Valeur approximative de delta_nu, en fonction du maximum d'autocorrelation 
	
	#error determination, cf equation (A.8) Mosser and Apourchaux (2009)
	beta = 0.763
	b = 5			#noise amplitude
	W_filter = N*delta_nu[ind]	#filter width	
	err_deltanu = (beta*b*deltanu**2)/(2*pi*Amax*W_filter)
	if graph[3]==1:
		if passage==1:
		    plt.figure(j+6)
		    plt.clf()
		    plt.title("Autocorrelation maxima vs frequency centering filters (1st calculation)")
		else:
		    plt.figure(j+14)
		    plt.clf()
		    plt.title("Autocorrelation maxima vs frequency centering filters (2nd calculation)")

		plt.plot(nu,maxi)
		plt.plot(nu,maxi, "ro", label=("Maxima autocorrelations"))
		#plt.plot(nu,tau/500000, "r", label=("tau") )
		plt.xlabel("frequency centering nuc [microHz]")
		plt.ylabel("Maxima + Fit")
		plt.legend()
	
	if graph[4]==1:
		if passage==1: 
		    plt.figure(j+7)
		    plt.clf()
		    plt.title('Autocorrelation maxima vs Delta nu (1st calculation)')
		else:	       
		    plt.figure(j+15)
		    plt.clf()
		    plt.title('Autocorrelation maxima vs Delta nu (2nd calculation)')

		plt.plot((2./tau)*1e6,maxi,'ro')
		#plt.plot(time_fit2, fit2, "g-", label=("Gauss Fit")) # data and the fit plot
		plt.xlabel("Delta nu [miHz]")
		plt.ylabel("Maxima")
		#plt.legend()

	""" Nu max determination """
	if (passage>=1) & fitnumax:	#if it is the second passage of ACF, we want to determine precisely numax by a fit

		try:	# on tente le fit:
			npt=500		#number of points for the gaussian fit
			c=5.		#standard deviation of the gaussian fit
			time_fit, fit, popt, pcov = mygauss_fit (nu,maxi,c,npt)	#gaussian fit to determine numax
			#time:time vector ; fit:result of gaussian fitting ; p1:best parameters found by least squart method
			#print "amplitude = ", popt[0], "+- ", sqrt(pcov[0][0])    
			#print "nu_max = ", popt[1], "+- ", sqrt(pcov[1][1]), " [muHz]"
			#print "Standard deviation = ", popt[2], "+- ", sqrt(pcov[2][2])
			#print "amplitude shift = ", popt[3], "+- ", sqrt(pcov[3][3])

		#Types d'erreur que l'on peut rencontrer:
		except RuntimeError as e:
			# une erreur de type 'runtime' a ete rencontree. On avertit l'utilisateur
			print "Runtime error in curve:"
			print e

			#On passe au resultat alternatif
			numax = nu[ind]		#Position du filtre pour lequel on obtient la valeur maximale de Amax
			err_numax = -1		#Pas d'erreur dans ce cas pour numax
			flag = 1		#Indique a l'utilisateur que le fit n'a pas converge.

		except ValueError as e:
			# une erreur de type 'runtime' a ete rencontree. On avertit l'utilisateur et l'on passe ensuite
			# aux instructions situees apres le 'else'
			print "ValueError"
			print e

			numax = nu[ind]		#Position du filtre pour lequel on obtient la valeur maximale de Amax
			err_numax = -1		#Pas d'erreur dans ce cas pour numax
			flag = 2		#Indique a l'utilisateur que le fit n'a pas converge.

		except:
			# une erreur inconnue a ete rencontree. On avertit  l'utilisateur et l'on fait remonter ('raise')
			# cette erreur, ce qui provoque l'arret du progamme
			print "Unexpected error", sys.exc_info()[0]
			raise

		else:
			# Si pas d'erreur, numax = resultat du fit:
			numax = popt[1]		#Resultat du fit: centre de la gaussienne
			err_numax = sqrt(pcov[1][1])	#erreur du fit sur numax
			flag = 0		#Indique a l'utilisateur que le fit a converge.

			if graph[3]==1:
			    plt.figure(j+6)
			    if passage==2: 
			        plt.figure(j+14)
			    plt.plot(time_fit, fit, "g-", label=("Gauss Fit")) # data and the fit plot
			    plt.legend()
	
	else:
		numax = nu[ind]
		err_numax = -1
		flag=-1		#Indique a l'utilisateur que le fit n'a pas ete fait


	return (Amax, numax, err_numax, deltanu, err_deltanu, flag) 


